/************************************************************************************
 * Copyright (C) 2002  Cristiano Sadun crsadun@tin.it
 *
 * You can redistribute this program and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation-
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ************************************************************************************/

package classUtils.pack.util.ant;

import classUtils.pack.util.CPoolReader;
import classUtils.putils.ClassPathBean;
import classUtils.putils.ClassPathUtils;
import collections.sortable.SortableVector;
import futils.DirList;
import gui.In;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.FileSet;
import utils.StringUtils;

import java.io.*;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

/**
 * An ant task to pack dependencies on a given set of classes.
 * <p/>
 * The available attributes are: <ul> <li><b>classes</b>: a comma-separated
 * list of classes to pack <li><b>packages</b>: a comma-separated list of
 * packages to pack. Each class in the package will be included, together
 * with its dependents. <li><b>classpath</b>: additional classpath to use
 * when packing classes (optional) <li><b>targetJar</b>: the name of the
 * jar file to produce <li><b>mainfestClasspath</b>: the (optional)
 * manifest Class-Path entry <li><b>mainfestMainclass</b>: the (optional)
 * manifest Main-Class entry <li><b>excludePkg</b>: a comma-separated list
 * of package prefixes to exclude. Defaults to <b>java,javax,sun</b>
 * <li><b>includePkg</b>: a comma-separated list of package prefixes to
 * include. Only classes in matching packages will be included. Has lower
 * precedence than excludePkg <li><b>resolveFiltered</b>: if <b>true</b>,
 * allows resolution of classes which are filtered out. Defaults to
 * <b>false</b>. <li><b>cacheClassFiles</b>: if <b>false</b>, disables
 * classfile caching - slower packing but saving memory </ul>
 * <p/>
 * Additional classpath can be also specified by a nested
 * <code>&lt;classpath&gt;</code> element.
 * <p/>
 * &lt;pack&gt; also supports inclusion of explicitly named additional
 * classes and/or files in the jar. Dependencies for such additional
 * classes will be computed and added too. This is done by declaring
 * internal <b>&lt;additionalclass&gt;</b> and <b>&lt;additionalfileset&gt;</b>
 * elements.
 * <p/>
 * <b>&lt;additionalclass&gt;</b> has a single <b>name</b> attribute which
 * contains the fully qualified name of the class to include. The class
 * must: <ul> <li> be in classpath; <li> not conflict with the filter
 * established by <b>excludePkg/includePkg</b>. </ul>
 * <p/>
 * For example,
 * <pre>
 * &lt;additionalclass name="javax.transaction.TransactionManager"/&gt;
 * </pre>
 * will add the <code>javax.transaction.TransactionManager</code> class and
 * all its dependent classes to the produced jar.
 * <p/>
 * <b>&lt;additionalfileset&gt;</b> is a standard Ant <code>FileSet</code>
 * structure which specifies a set of files to unconditionally add to the
 * produced jar.
 * <p/>
 * For example,
 * <pre>
 *   &lt;additionalfileset dir="${basedir}"&gt;
 * 	  &lt;include name="META-INF/services/*"/&gt;
 *   &lt;/additionalfileset&gt;
 * </pre>
 * <p/>
 * will add any file under the <code>META-INF/service</code> subdirectory
 * of the current <code>${basedir}</code> directory.
 * <p/>
 * <p/>
 * $Revision$
 *
 * @author Cristiano Sadun
 * @version 1.6
 */
public class Pack {
    private ClassMap clsMap;

    private final PackTask task = new PackTask(this);
    private ClassFinderUtils classFinderUtils;

    private ClassPathBean cpb = ClassPathBean.restore();

    public Pack() {
        init();
    }

    private void init() {
        classFinderUtils =
                new ClassFinderUtils(getClassPath());
        clsMap = new ClassMap();
    }


    private String classes;
    private String packages;
    private String targetJar;
    private boolean resolveFiltered = false;
    private String excludePkg;
    private String includePkg;

    private String manifestClassPath;
    private String manifestMainClass;
    private boolean hasCacheClassFiles = true;
    private boolean detectrmi = false;

    private ClassFilter filter;
    private HashSet refusedNames;
    private Set additionalFiles = new HashSet();
    private Set additionalClasses = new HashSet();
    private Set resources = new HashSet();

    private Set ignorableClasses = new HashSet();

    void writeOutJar() {
        try {
            JarOutputStream jos = new JarOutputStream(new BufferedOutputStream(
                    new FileOutputStream(targetJar)));
            if (manifestClassPath != null |
                    manifestMainClass != null) {
                log("Creating manifest");
                Manifest manifest = new Manifest();
                manifest.getMainAttributes().put(
                        Attributes.Name.MANIFEST_VERSION,
                        "1.0");
                if (manifestClassPath != null) {
                    manifest.getMainAttributes()
                            .put(Attributes.Name.CLASS_PATH,
                                    manifestClassPath);
                }
                if (manifestMainClass != null) {
                    manifest.getMainAttributes()
                            .put(Attributes.Name.MAIN_CLASS,
                                    manifestMainClass);
                }
                JarEntry entry = new JarEntry("META-INF/MANIFEST.MF");
                jos.putNextEntry(entry);
                manifest.write(jos);
            }

            log("Packing " + targetJar);

            processDependencies(jos);

            // Also, determine and add additional files
            if (additionalFiles.size() > 0) {
                for (Iterator i = additionalFiles.iterator();
                     i.hasNext();) {
                    FileSet fs = (FileSet) i.next();
                    DirectoryScanner ds = fs.getDirectoryScanner(
                            task.getProject());
                    ds.scan();
                    String[] includedFiles = ds.getIncludedFiles();
                    for (int j = 0; j <
                            includedFiles.length; j++) {
                        File f = new File(ds.getBasedir() +
                                File.separator +
                                includedFiles[j]);
                        log("Adding file " +
                                includedFiles[j],
                                Project.MSG_VERBOSE);
                        // Let's the jar entry have the same name as the file, minus the base directory
                        String fl = StringUtils.replaceAllSb(
                                includedFiles[j],
                                File.separator,
                                "/");
                        JarEntry entry = new JarEntry(fl);
                        jos.putNextEntry(entry);
                        InputStream is = new BufferedInputStream(
                                new FileInputStream(f));
                        int c;
                        while ((c = is.read()) !=
                                -1)
                            jos.write(c);
                    }
                }
            }

            // And resources
            for (Iterator i = resources.iterator(); i.hasNext();) {
                Resource res = (Resource) i.next();
                log("Adding resource " + res,
                        Project.MSG_VERBOSE);
                InputStream is = classFinderUtils.openResource(res.name);
                if (is == null)
                    throw new BuildException("resource " +
                            res.name +
                            " not found. ClassPath is " +
                            classFinderUtils.getClassPath());
                JarEntry entry = new JarEntry(res.name);
                jos.putNextEntry(entry);
                int c;
                while ((c = is.read()) != -1)
                    jos.write(c);
            }

            jos.close();
        } catch (IOException e) {
            e.printStackTrace();
            throw new BuildException(e);
        }
    }

    void addAdditionalClasses() {
        try {
            for (Iterator i = additionalClasses.iterator(); i.hasNext();) {
                ClassSpec cls = (ClassSpec) i.next();
                log("Finding dependencies for additional class " +
                        cls.name,
                        Project.MSG_VERBOSE);
                // Find the dependencies for each additional class
                findDependencies(this, classFinderUtils, detectrmi, filter, refusedNames, resolveFiltered, cls.name,
                        clsMap,
                        true);
            }
        } catch (ClassNotFoundException e) {
            log("The current class path is " +
                    classFinderUtils.getClassPath(),
                    Project.MSG_ERR);
            throw new BuildException(e);
        } catch (IOException e) {
            throw new BuildException(e);
        }
    }

    void computeDependencies(String[] clsNames) throws IOException,
                                                       ClassNotFoundException {
        for (int i = 0; i < clsNames.length; i++)
            listDependencies(clsNames[i]);
    }

    private void listDependencies(final String clsName)
            throws IOException, ClassNotFoundException {
        log("Calculating dependencies for " +
                clsName);
        log("Classpath is " +
                classFinderUtils.getClassPath(),
                Project.MSG_VERBOSE);
        // Find the dependecies for each class
        findDependencies(this, clsName,
                clsMap);
    }

    public void validateInput() {
        if (targetJar == null)
            throw new BuildException(
                    "Missing mandatory targetJar attribute");
        if (classes == null && packages == null)
            throw new BuildException(
                    "Missing mandatory classes or packages attribute");
        if (classes != null && packages != null)
            throw new BuildException(
                    "Only one of classes or packages can be specified");


    }

    public void setIgnorableClasses() {
        Set hs = new HashSet();
        for (Iterator i = ignorableClasses.iterator(); i.hasNext();) {
            hs.add(((ClassSpec) i.next()).name);
        }
        ignorableClasses = hs;
    }

    private void processDependencies(JarOutputStream jos)
            throws IOException {
        for (Iterator i = clsMap.getKeySet()
                .iterator(); i.hasNext();) {
            String clsName = (String) i.next();
            String entryName = clsName.replace('.', '/') +
                    ".class";
            JarEntry entry = new JarEntry(entryName);
            jos.putNextEntry(entry);
            byte[] bytecode = (byte[]) clsMap.get(clsName);
            ByteArrayInputStream is = new ByteArrayInputStream(bytecode);
            int c;
            while ((c = is.read()) != -1)
                jos.write(c);
        }
    }

    public void printDependencies() throws Exception {
        String cn = In.getString("enter class name");
        if (cn == null) return;
        printDependencies(cn);
    }

    private void printDependencies(String cn) throws Exception {
        setManifestMainClass(cn);
        setClasses(cn);

        final String targetJar = cn + ".jar";
        In.message("am creating output file:" + targetJar);
        setTargetJar(targetJar);
        try {
            packIt();
        } catch (BuildException e) {
            In.message(e + " for class:" + cn);
            printDependencies(cn);
        }
        printDependcies();
    }

    private void printDependcies() {
        SortableVector sv = new SortableVector();
        if (sv == null) return;
        if (clsMap == null) return;
        for (Iterator i = clsMap.getKeySet()
                .iterator(); i.hasNext();) {
            String clsName = (String) i.next();
            sv.addElement(clsName);

        }
        sv.sort();
        log("Dependencies:" + sv.size());
        for (int i = 0; i < sv.size(); i++)
            log(sv.elementAt(i).toString());
    }


    public static void findDependencies(Pack pack, String clsName,
                                        ClassMap clsMap)
            throws IOException, ClassNotFoundException {
        findDependencies(pack,
                pack.classFinderUtils,
                pack.detectrmi,
                pack.filter,
                pack.refusedNames,
                pack.resolveFiltered,
                clsName,
                clsMap,
                false);
    }

    /**
     * The heart of the file finder. This fails in Java Webstart because I
     * cannot control how the path is set for class searches. todo fix this
     * bug!
     *
     * @param pack
     * @param classFinderUtils1
     * @param detectrmi1
     * @param filter1
     * @param refusedNames1
     * @param resolveFiltered1
     * @param clsName
     * @param clsMap
     * @param failOnUnaccepted
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public static void findDependencies(Pack pack,
                                        ClassFinderUtils classFinderUtils1,
                                        boolean detectrmi1,
                                        ClassFilter filter1,
                                        HashSet refusedNames1,
                                        boolean resolveFiltered1,
                                        String clsName,
                                        ClassMap clsMap,
                                        boolean failOnUnaccepted)
            throws IOException,
                   ClassNotFoundException {

        if (pack.ignorableClasses.contains(clsName)) {
            pack.log(clsName +
                    " ignored as configured",
                    Project.MSG_VERBOSE);
            return;
        }

        if (clsMap.contains(clsName)) {
            //log(clsName+" already accepted.", project.MSG_VERBOSE);
            return;
        }
        if (refusedNames1.contains(clsName)) {
            //log(clsName+" already refused.", project.MSG_VERBOSE);
            return;
        }

        // Is the name an array? Try to find the component class and return
        if (clsName.startsWith("[L")) {
            String clsName2 = clsName.substring(2, clsName.length() - 1);
            Pack.findDependencies(pack, clsName2, clsMap);
            return;
        }
        if (clsName.startsWith("[")) {
            String clsName2 = clsName.substring(1);
            if ("B".equals(clsName2))
                return;
            else if ("C".equals(clsName2))
                return;
            else if ("D".equals(clsName2))
                return;
            else if ("F".equals(clsName2))
                return;
            else if ("I".equals(clsName2))
                return;
            else if ("J".equals(clsName2))
                return;
            else if ("S".equals(clsName2))
                return;
            else if ("Z".equals(clsName2))
                return;
            else if ("V".equals(clsName2)) return;
            Pack.findDependencies(pack, clsName2, clsMap);
            return;
        }

        // Load the class
        byte[] bytecode = classFinderUtils1.getClassBytes(clsName);
        CPoolReader.ClassFile cf = classFinderUtils1.readClassData(
                bytecode);
        // Add it to the set, if not filtered out
        String[] tmp = ClassPathUtils.splitClassName(clsName);
        boolean accepted = filter1.accept(tmp[0],
                tmp[1],
                cf);
        if (failOnUnaccepted && !accepted)
            throw new BuildException("The class " +
                    tmp[0] +
                    "." +
                    tmp[1] +
                    " is not acceptable with the current " +
                    "includePkg/excludePkg settings (" +
                    filter1 +
                    ")");
        if (accepted) {
            // todo find out how this works
            clsMap.put(clsName, bytecode);
            pack.log(clsName + " accepted.",
                    Project.MSG_VERBOSE);
        } else {
            refusedNames1.add(clsName);
            pack.log(clsName + " refused.",
                    Project.MSG_VERBOSE);
        }
        if (accepted || resolveFiltered1) {
            // If RMI detection is active and the
            // class implements UnicastRemoteObject,
            // try to find the stubs
            if (detectrmi1 && cf.isInterface()) {
                String superClass = cf.getSuperClass();
                if (superClass.equals("java.rmi.Remote")) {
                    String stubClsName = clsName +
                            "_Stub";
                    byte[] stubBytecode = classFinderUtils1.getClassBytes(
                            stubClsName);
                    clsMap.put(stubClsName,
                            stubBytecode);
                }
            }

            // Browse trhu all the class names mentioned in
            // the constant pool, and find all their dependencies
            String[] usedClasses = cf.getUsedClasses();
            for (int i = 0; i <
                    usedClasses.length; i++) {
                String usedClassName = usedClasses[i].replace('/', '.');
                Pack.findDependencies(pack, usedClassName,
                        clsMap);
            }
        }
    }


    /**
     * Returns the classes.
     *
     * @return String
     */
    public String getClasses() {
        return classes;
    }

    /**
     * Sets the classes.
     *
     * @param classes The classes to set
     */
    public void setClasses(String classes) {
        this.classes = classes;
    }

    /**
     * Returns the targetJar.
     *
     * @return String
     */
    public String getTargetJar() {
        return targetJar;
    }

    /**
     * Sets the targetJar.
     *
     * @param targetJar The targetJar to set
     */
    public void setTargetJar(String targetJar) {
        this.targetJar = targetJar;
    }

    /**
     * Returns the resolveFiltered.
     *
     * @return boolean
     */
    public boolean getResolveFiltered() {
        return resolveFiltered;
    }

    /**
     * Sets the resolveFiltered.
     *
     * @param resolveFiltered The resolveFiltered to set
     */
    public void setResolveFiltered(boolean resolveFiltered) {
        this.resolveFiltered = resolveFiltered;
    }

    /**
     * Returns the excludePkg.
     *
     * @return String
     */
    public String getExcludePkg() {
        return excludePkg;
    }

    /**
     * Sets the excludePkg.
     *
     * @param excludePkg The excludePkg to set
     */
    public void setExcludePkg(String excludePkg) {
        this.excludePkg = excludePkg;
    }

    /**
     * Returns the classpath.
     *
     * @return String
     */
    public String getClassPath() {
        return cpb.getClassPath();
    }

    /**
     * Returns the manifestClassPath.
     *
     * @return String
     */
    public String getManifestClassPath() {
        return manifestClassPath;
    }

    /**
     * Returns the manifestMainClass.
     *
     * @return String
     */
    public String getManifestMainClass() {
        return manifestMainClass;
    }

    /**
     * Sets the manifestClassPath.
     *
     * @param manifestClassPath The manifestClassPath to set
     */
    public void setManifestClassPath(String manifestClassPath) {
        this.manifestClassPath =
                manifestClassPath;
    }

    /**
     * Sets the manifestMainClass.
     *
     * @param manifestMainClass The manifestMainClass to set
     */
    public void setManifestMainClass(String manifestMainClass) {
        this.manifestMainClass =
                manifestMainClass;
    }

    /**
     * Returns the includePkg.
     *
     * @return String
     */
    public String getIncludePkg() {
        return includePkg;
    }

    /**
     * Sets the includePkg.
     *
     * @param includePkg The includePkg to set
     */
    public void setIncludePkg(String includePkg) {
        this.includePkg = includePkg;
    }

    /**
     * Ant entry point for <code>additionalfileset</code> subelements.
     * <p/>
     *
     * @param fs the fileset object to add, created by Ant engine
     */
    public void addAdditionalFileSet(FileSet fs) {
        additionalFiles.add(fs);
    }
    /**
     * This is an untested means of addition additional
     * files to a packer.
     *  - DocJava.
     * @param f
     */
    public void addAdditionalFiles(File f[]) {
        FileSet fs = new FileSet();
        for (int i = 0; i < f.length; i++)
            fs.setIncludesfile(f[i]);
        addAdditionalFileSet(fs);
    }
    /**
     * An experimental way to add a list of files
     * to a packed output Jar. This might help with
     * resource bundles.
     *  - DocJava.
     * @param dl
     */
    public void addAddtionalFiles(DirList dl){
        addAdditionalFiles(dl.getFiles());
    }

    /**
     * Ant entry point for <code>additionalClass</code> subelements.
     * <p/>
     *
     * @return AdditionalClass an object containing info about the class to
     *         add
     */
    public ClassSpec createAdditionalClass() {
        ClassSpec cls = new ClassSpec();
        additionalClasses.add(cls);
        return cls;
    }

    /**
     * Ant entry point for <code>ignoreClass</code> subelements.
     * <p/>
     *
     * @return AdditionalClass an object containing info about the class to
     *         add
     */
    public ClassSpec createIgnoreClass() {
        ClassSpec cls = new ClassSpec();
        ignorableClasses.add(cls);
        return cls;
    }

    /**
     * Ant entry point for <code>additionalClass</code> subelements.
     * <p/>
     *
     * @return AdditionalClass an object containing info about the class to
     *         add
     */
    public Resource createResource() {
        Resource res = new Resource();
        resources.add(res);
        return res;
    }

    /**
     * Returns the cacheClassFiles.
     *
     * @return boolean
     */
    public boolean hasCacheClassFiles() {
        return hasCacheClassFiles;
    }

    /**
     * Sets the cacheClassFiles.
     *
     * @param hasCacheClassFiles The cacheClassFiles to set
     */
    public void setHasCacheClassFiles(boolean hasCacheClassFiles) {
        this.hasCacheClassFiles = hasCacheClassFiles;
    }

    /**
     * Returns the packages.
     *
     * @return String
     */
    public String getPackages() {
        return packages;
    }

    /**
     * Sets the packages.
     *
     * @param packages The packages to set
     */
    public void setPackages(String packages) {
        this.packages = packages;
    }

    public void packIt() throws IOException, ClassNotFoundException {
        task.packIt();
    }

    public void log(String s) {
        task.log(s);
    }

    public void log(String s, int i) {
        task.log(s, i);
    }

    void setRefusedNames(HashSet refusedNames) {
        this.refusedNames = refusedNames;
    }

    void setFilter(ClassFilter filter) {
        this.filter = filter;
    }

    void setCache(boolean b) {
        classFinderUtils.setCache(true);
    }
}
