package classUtils.pack.util;

import java.util.*;

/**
 * Execute the same action for each object in a collection, optionally filtering the objects by the given filter.
 * <p>
 * By default, ForEach doesn't do anything, so the typical way to employ ForEach is by subclassing anonymously
 * overriding the method <a href="#action(java.lang.Object)>action()</a>.  For example
 * <p><pre>
 * List list;
 *  ...<i>create and populate list</i>...
 * new ForEach(l) {
 *   public void action(Object obj) {
 *      System.out.println(obj);
 *	  }
 * }
 * </pre>
 * prints all the elements of a List collection.
 * <p>
 * The default implementation of <a href="#action(java.lang.Object)>action()</a> does exactly this.

 */
public class ForEach {

	/**
	 * Users can implement this interface (anonymously or via a named class) to apply the action
	 * only to a subset of the objects in the collection.
	 */
	public interface Filter {
		/**
		 * Determines if the action is to be applied to the object or not
		 * @param obj the object to verify
		 * @return <b>true</b> if the action must be applied to the object
		 */
		public boolean accept(Object obj);
	}

	/**
	 * A Null filter, which doesn't filter any object
	 */
	private static class NullFilter implements Filter {
		/**
		 * Returns always true
		 * @param obj the object to verify
		 * @return <b>true</b> always
		 */
		public boolean accept(Object obj) { return true; }
	}

	private Iterator i;
	private Filter filter;
	private boolean atStart=true;

	/**
	 * Creates a ForEach object applying to the elements of the given collection satisfying the given filter
	 * @param c the collection on whose elements which the action will be executed
	 * @param filter the filter to apply
	 */
	public ForEach(Collection c, Filter filter) {
		this(c.iterator(), filter);
	}

	/**
	 * Creates a ForEach object applying to all the elements of the given collection
	 * @param c the collection on whose elements which the action will be executed
	 */
	public ForEach(Collection c) {
		this(c, null);
	}

	/**
	 * Creates a ForEach object applying to the elements of the given array satisfying the given filter
	 * @param array the array on whose elements which the action will be executed
	 * @param filter the filter to apply
	 */
	public ForEach(Object [] array, Filter filter) {
		this(Arrays.asList(array), filter);
	}

	/**
	 * Creates a ForEach object applying to all the elements of the given array
	 * @param array the array on whose elements which the action will be executed
	 */
	public ForEach(Object [] array) {
		this(array, null);
	}

	/**
	 * Creates a ForEach object applying to the elements of the given iterator satisfying the given filter
	 * @param i an iterator, on whose elements which the action will be executed
	 * @param filter the filter to apply
	 */
	public ForEach(Iterator i, Filter filter) {
		if (i==null) throw new IllegalArgumentException("Collection can't be 'null'");
		if (filter==null) filter=new NullFilter();
		this.i=i;
		this.filter=filter;
	}

	/**
	 * Creates a ForEach object applying to all the elements of the given iterator
	 * @param i an iterator, on whose elements which the action will be executed
	 */
	public ForEach(Iterator i) {
		this(i, null);
	}

	/**
	 * Executes the action of the elements of the collection/iterator. If a filter has been defined,
	 * the action is executed only on the elements satisfying the filter.
	 *
	 */
	public void execute() {
		if (! atStart) throw new IllegalStateException("execute() already invoked: use reInit() before reinvoking");
		for(;i.hasNext();) {
			Object obj = i.next();
			if (filter.accept(obj)) action(obj);
		}
		atStart=false;
	}

	/**
	 * Re-initializes the ForEach object on the given collection
	 * @param Collection c the collection on whose elements which the action will be executed
	 */
	public void reInit(Collection c) {
		reInit(c.iterator());
	}

	/**
	 * Re-initializes the ForEach object on the given iterator
	 * @param Iterator i an iterator, on whose elements which the action will be executed
	 */
	public void reInit(Iterator i) {
		this.i=i;
		atStart=true;
	}


	/**
	 * This method must be overridden by a subclass to define the actual action.
	 * <p>By default, the object's toString() method is invoked and the result
	 * printed on System.out.
	 */
	public void action(Object obj) {
		System.out.println(obj);
	}


}