package classUtils.pack.util;

import classUtils.putils.ClassPathUtils;
import classUtils.putils.ClassPathBean;
import classUtils.javassist.NotFoundException;
import classUtils.javassist.CannotCompileException;
import gui.In;
import gui.JInfoFrame;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;

/**
 * A classloader whose classpath can be set dynamically.
 * <p/>
 * Version 1.2 adds resource support capability, but for the {@link
 * #findResources(java.lang.String) findResources()} which is not
 * implemented.
 * <p/>
 * By default, the classLoader attempts to use the parent class loader, if
 * any, to find a class. By setting the {@link #setForceDynamicLoading(boolean)
 * forceDynamicLoading} property, the classLoader will always find classes
 * by using the embedded {@link classUtils.pack.util.DynamicClassFileFinder
 * DynamicClassFileFinder}.
 * <p/>
 * $Revision$
 *
 * @author cris
 * @version 1.3
 */
public class DynamicClassLoader extends ClassLoader {

    private DynamicClassFileFinder cff;
    private DynamicResourceFileFinder rff;
    private boolean forceDynamicLoading;

    /**
     * Constructor for DynamicClassLoader.
     *
     * @param arg0
     */
    public DynamicClassLoader(ClassLoader arg0,
                              DynamicClassFileFinder cff,
                              DynamicResourceFileFinder rff) {
        super(arg0);
        this.cff = cff;
        this.rff = rff;
    }

    /**
     * Constructor for DynamicClassLoader.
     */
    public DynamicClassLoader(DynamicClassFileFinder cff,
                              DynamicResourceFileFinder rff) {
        super();
        this.cff = cff;
        this.rff = rff;
    }

    public static void main(String[] args)  {
        try {
            DynamicClassLoader dcl = new DynamicClassLoader();
            JInfoFrame jif = new JInfoFrame();
            jif.setSize(200, 200);
            jif.setVisible(true);
            DynamicClassFileFinder dcff = dcl.getClassFileFinder();
            dcff.setClassPath(ClassPathUtils.getClassPath());
            ClassPathUtils.getClassFromFile();
        } catch (NotFoundException e) {
            In.message(e);

        } catch (IOException e) {
            In.message(e);

        } catch (CannotCompileException e) {
            In.message(e);

        }
    }

    private static DynamicJDK12ClassFileFinder dynamicJDK12ClassFileFinder =
            new DynamicJDK12ClassFileFinder(ClassPathBean.restore());

    public DynamicClassFileFinder getClassFileFinder() {
        return cff;
    }

    public DynamicResourceFileFinder getResourceFileFinder() {
        return rff;
    }


    public DynamicClassLoader(ClassLoader arg0) {
        this(arg0,
                dynamicJDK12ClassFileFinder,
                dynamicJDK12ClassFileFinder);
    }

    /**
     * Add a class path entry.
     *
     * @param entry
     */
    public void addClassPathEntry(String entry) {
        synchronized (this) {
            cff.addClassPathEntry(entry);
            rff.addClassPathEntry(entry);
        }
    }

    /**
     * Set the class path.
     *
     * @param classPath
     */


    /**
     * Constructor for DynamicClassLoader.
     */
    public DynamicClassLoader() {
        this(dynamicJDK12ClassFileFinder, dynamicJDK12ClassFileFinder);
    }


    /**
     * @see java.lang.ClassLoader#findClass(java.lang.String)
     */
    public Class findClass(String arg0) throws ClassNotFoundException {
        if (arg0 == null) return null;
        //System.out.println("Dynamic class loader finding class "+arg0);
        try {
            if (forceDynamicLoading) throw new ClassNotFoundException();
            return super.findClass(arg0);
        } catch (ClassNotFoundException e) {
            try {
                //System.out.println("Searching for class "+ arg0 + "; classpath is "+cff.getClassPath());
                In.message("Searching for class:" + arg0);
                byte[] data = cff.getClassBytes(arg0);
                return this.defineClass(arg0, data, 0, data.length);
            } catch (IOException e2) {
                e2.printStackTrace();
                throw new ClassNotFoundException(
                        "IOException while reading definition for class " +
                        arg0,
                        e2);
            }
        }
    }


    /**
     * @see java.lang.ClassLoader#findResource(java.lang.String)
     */
    protected URL findResource(String name) {
        try {
            File f = rff.findResourceFile(name);
            if (f == null) return null;
            URL fileURL = new URL("file",
                    null,
                    rff.findResourceFile(name).getCanonicalPath());
            if (f.equals(new File(name))) // It's the resource itself
                return fileURL;
            else
                return new URL("jar:" + fileURL + "!" + name);
        } catch (MalformedURLException e) {
            return null;
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * @see java.lang.ClassLoader#findResources(java.lang.String)
     */
    protected Enumeration findResources(String name) throws IOException {
        throw new RuntimeException(
                "findResources is not implemented by this ClassLoader");
    }

    /**
     * @see java.lang.ClassLoader#getResourceAsStream(java.lang.String)
     */
    public InputStream getResourceAsStream(String name) {
        InputStream is;
        if ((is = super.getResourceAsStream(name)) != null) return is;
        try {
            return rff.openResource(name);
        } catch (IOException e) {
            return null;
        }
    }

    /*
     * @see java.lang.ClassLoader#getResource(java.lang.String)
     *
    public URL getResource(String name) {
        URL url=null;
        if (getParent() != null) {
            url = getParent().getResource(name);
        } else {
            url = findResource(name);
        }
        return url;
    }
    */


    /**
     * Returns the forceDynamicLoading.
     *
     * @return boolean
     */
    public boolean isForceDynamicLoading() {
        return forceDynamicLoading;
    }

    /**
     * Sets the forceDynamicLoading.
     *
     * @param forceDynamicLoading The forceDynamicLoading to set
     */
    public void setForceDynamicLoading(boolean forceDynamicLoading) {
        this.forceDynamicLoading = forceDynamicLoading;
    }

    /**
     * @see java.lang.ClassLoader#loadClass(java.lang.String, boolean)
     */
    protected synchronized Class loadClass(String name, boolean resolve)
            throws ClassNotFoundException {
        if (forceDynamicLoading && !name.startsWith("java.")) {
            Class c;
            if ((c = super.findLoadedClass(name)) != null) {
                //System.out.println(name+" already loaded");
                return c;
            }
            //System.out.println(name+" not loaded yet");
            return findClass(name);
        } else
            return super.loadClass(name, resolve);
    }


}
