package bookExamples.ch27BusinessGraphics.charts;

import java.awt.*;

/**
 * DrawUtil is the drawing utility class which draws the actual
 * graphs and their labels. It can only be constructed with an
 * instance of DoubleDataBean. It is invoked by each type of graphics.graph.
 * Global Variables:
 * <UL>
 * <LI> DoubleDataBean: instance of DoubleDataBean so the graphics.graph is drawn
 *	   with the proper data
 * <LI> width: from DoubleDataBean, the width of the image
 * <LI> height: from DoubleDataBean, the height of the image
 * </UL>
 *
 * @author	Allison McHenry
 * @author	Douglas Lyon, PhD
 * @since	JDK 1.3
 */

public class DrawUtil {

//GLOBAL VARIABLES
    private DoubleDataBean dd = null;
    private int width = 0;
    private int height = 0;


//CONSTRUCTOR
    /**
     *	Default constructor, used to instantiate an instance of DrawUtil by
     *	the LineGraph, PieGraph and BarGraph classes
     *	@param dd	Instance of DoubleDataBean containing data to be drawn
     */
    protected DrawUtil(DoubleDataBean _dd) {
        dd = _dd;
        width = dd.getWidth();
        height = dd.getHeight();
    }


    /**
     *	Draws the title of the X axis for line graphics.graph and bar graphics.graph,
     *	in the center of the axis.
     *
     *	@param g 	Graphics context for drawing
     *	@see		LineGraph.#paint
     *	@see		BarGraph.#paint
     *	@see		DoubleDataBean.#getYAxisCoord
     *	@see		java.awt.Graphics.#drawString
     *	@see		java.awt.Graphics.#FontMetrics
     *	@see		java.awt.Graphics.#setFont
     */
    protected void drawXAxisLabel(Graphics g) {
        String xLabel = dd.getXLabel();
        double yOrigin = dd.getYAxisCoord();

        g.setColor(Color.black);
        g.setFont(new Font("SansSerif", Font.PLAIN, 12));
        FontMetrics fm = g.getFontMetrics();

        int xStringStartX = ((width / 2) - (fm.stringWidth(xLabel) / 2));
        int xStringStartY = (int) (yOrigin + (2 * dd.Y_OFFSET));
        g.drawString(xLabel, xStringStartX, xStringStartY);
    }

    /**
     *	Draws the title of the Y axis for line graphics.graph and bar graphics.graph,
     *	in the upper left corner of the graphics.graph.
     *
     *	@param g 	Graphics context for drawing
     *	@see		LineGraph.#paint
     *	@see		BarGraph.#paint
     *	@see		DoubleDataBean.#getXAxisCoord
     *	@see		java.awt.Graphics.#drawString
     */

    protected void drawYAxisLabel(Graphics g) {
        String yLabel = dd.getYLabel();
        double xAxisCoord = dd.getXAxisCoord();

        g.setColor(Color.black);
        g.setFont(new Font("SansSerif", Font.PLAIN, 12));
        FontMetrics fm = g.getFontMetrics();
        int yStringStartX = (int) (xAxisCoord + dd.X_OFFSET);
        int yStringStartY = 4 * dd.Y_OFFSET;

        if (xAxisCoord == (dd.getWidth() - dd.X_OFFSET)) {
            yStringStartX = (int) (xAxisCoord - fm.stringWidth(yLabel));
            yStringStartY = 4 * dd.Y_OFFSET;
        }
        g.drawString(yLabel, yStringStartX, yStringStartY);
    }

    /**
     *	Draws the title of the graphics.graph for line graphics.graph and bar graphics.graph,
     *	in the bottom center of the graphics.graph.
     *
     *	@param g 	Graphics context for drawing
     *	@see		LineGraph.#paint
     *	@see		BarGraph.#paint
     *	@see		java.awt.Graphics.#drawString
     *	@see		java.awt.Graphics.#FontMetrics
     *	@see		java.awt.Graphics.#setFont
     */
    protected void drawTitle(Graphics g) {
        String title = dd.getTitle();

        g.setColor(Color.yellow);
        g.setFont(new Font("SansSerif", Font.BOLD, 16));
        FontMetrics fontM = g.getFontMetrics();

        int titleStringStartX = ((width / 2) - (fontM.stringWidth(title) / 2));
        int titleStringStartY = height - dd.Y_OFFSET;
        g.drawString(title, titleStringStartX, titleStringStartY);
    }

    /**
     *	Draws the title of the graphics.graph for line graphics.graph and bar graphics.graph,
     *	at the location specified as (x,y)
     *
     *	@param g 		Graphics context for drawing
     *	@param xCoord 	x coordinate for where the string should be drawn
     *	@param yCoord 	y coordinate for where the string should be drawn
     *	@see			PieGraph.#paint
     *	@see			java.awt.Graphics.#drawString
     *	@see			java.awt.Graphics.#FontMetrics
     *	@see			java.awt.Graphics.#setFont
     */
    protected void drawTitle(Graphics g, int xCoord, int yCoord) {
        String title = dd.getTitle();

        g.setColor(Color.yellow);
        g.setFont(new Font("SansSerif", Font.BOLD, 16));
        FontMetrics fontM = g.getFontMetrics();

        int titleStringStartX = xCoord;
        int titleStringStartY = yCoord;
        g.drawString(title, titleStringStartX, titleStringStartY);
    }
}