package serialPorts;

import gnu.io.*;

import javax.comm.SerialPort;
import javax.comm.UnsupportedCommOperationException;
import javax.comm.PortInUseException;
import javax.comm.CommPortIdentifier;
import javax.comm.CommPort;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;

public class ModemPort {


    /** holds the using application name - is good for sorting conflicts */
    private String applicationName = "Modem Through Java Application";
    /** How long to wait for the open to finish up. */
    private int timeOutInSeconds = 30;
    /** The chosen Port itself */
    private CommPort commPort;
    /** the requested port that the modem reside at this should be set as a string
     * in the format COMx and the x stand for the port number
     **/
    private String portName = "/dev/cu.modem";
    /** The baud rate to use. */
    private int baud = 9600;
    /** will hold the current single port as we loop hrough all available in the system */
    private CommPortIdentifier cpid = null;
    /** this holds the message we want to send to the modem */
    private String messageString = "atdt8770890";
    /** this will hold the serial port choosen to be used */
    private SerialPort serialPort = null;
    /** this object is used to send data to the serial port */
    private PrintStream ps = null;
    /** this object is used to receive data from the serial port */
    private BufferedReader inputStream;


    public ModemPort() {
    }

    public ModemPort(String portName,
                     String messageToSend) {
        this.portName = portName;
        messageString = messageToSend;
    }

    /**
     * this method will open the port and prepare it for use
     */
    public boolean openPort() {
        boolean returnedValue = true;

        cpid = Utils.getPortByName(portName);
        try {
            returnedValue = initPort(returnedValue);
        } catch (PortInUseException err) {
            System.out.println(
                    "this port is in use by another application named: "
                    + err.currentOwner
                    + " please release it before running again");
            returnedValue = false;
        } catch (UnsupportedCommOperationException err) {
            System.out.println(
                    "you have tried to set a"
                    + " property to a port that does not"
                    + " support this property maybe you are"
                    + " using paralel port instead of serial");
            returnedValue = false;
        }
        return returnedValue;
    }

    private boolean initPort(boolean returnedValue) throws
            PortInUseException,
            UnsupportedCommOperationException {
        // open the choosen port and set the
        // time to wait for it to open also state the opening application name
        commPort = cpid.open(applicationName, timeOutInSeconds * 1000);
        serialPort = (SerialPort) commPort;
        //set the serial port properties
        serialPort.setDTR(true);
        serialPort.setRTS(true);
        serialPort.setSerialPortParams(baud, SerialPort.DATABITS_8, SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);
        try {
            initStreams();
        } catch (IOException err) {
            System.out.println("could not get the ports output and input streams");
            returnedValue = false;
        }
        return returnedValue;
    }

    private void initStreams() throws IOException {
        //get the output stream from the serial port
        //object and then create a new print stream object with it
        ps = new PrintStream(
                serialPort.getOutputStream());
        //get the input stream from the serial port object
        //and then create a new BufferedReader object with it
        inputStream = new BufferedReader(
                new InputStreamReader(
                        serialPort.getInputStream()));
    }

    /**this method will send the current set message to the serial and make sure
     * that the modem echoed the command back, also the modem will send the prompt
     * line which this method will read and discard it */
    public boolean sendToPort(String whatToSend) {
        boolean returnedValue = true;

        try {
            sendMessage(whatToSend);

        } catch (IOException err) {
            System.out.println("error occurred while trying to send data to the serial port through the print writer object");
            returnedValue = false;
        }
        return returnedValue;
    }

    private void sendMessage(String whatToSend) throws IOException {
        String junk = "";
        //first set the string to send
        setMessageString(whatToSend);
        //System.out.println("user> " + whatToSend);
        ps.print(messageString);
        ps.print("\r\n");
        //Expect the modem to echo the command we
        // have just sent if not then notify it
        if (!receive(messageString)) {
            System.err.println("WARNING: Modem did not echo command.");
        }
        //The modem sends an extra blank line by way of a prompt.
        //Here we read and discard it.
        junk = inputStream.readLine();
        if (junk.length() != 0) {
            System.err.print("Warning unexpected response: ");
            System.err.println(junk);
        }
    }

    /** same method as the above only allowing to preset the message */
    public boolean sendToPort() {
        return sendToPort(messageString);
    }

    /** this method will receive an expected result from the serial port and reply
     * if the received data is the expected or not
     **/
    public boolean receive(String whatToExpect) {
        try {
            return isThisReturnedValue(whatToExpect);
        } catch (IOException err) {
            System.out.println("error occurred while "
                    +" to receive data from the "
                    +"serial port through "
                    +"the reader object");
        }

        return false;
    }

    private boolean isThisReturnedValue(String whatToExpect)
            throws IOException {
        String response = inputStream.readLine();
        System.out.println("modem> " + response);
        return response.indexOf(whatToExpect) >= 0;
    }

    /** this method will close the writing and reading objects*/
    public boolean closePort() {
        boolean returnedValue = true;
        //first rest so the modem can finish its job
        try {
            Thread.sleep(5000);
        } catch (InterruptedException e) {
        }


        //then close the port
        try {
            inputStream.close();
            ps.close();
            serialPort.close();
        } catch (IOException err) {
            System.out.println(
                    "error occurred while trying to close"
                    +"the reader and writer objects");
            returnedValue = false;
        }
        return returnedValue;
    }

    public void printLightStatus() {
        boolean CDStatus = serialPort.isCD();
        boolean CTSStatus = serialPort.isCTS();
        boolean DSRStatus = serialPort.isDSR();
        boolean DTRStatus = serialPort.isDTR();
        System.out.println(
                "CDStatus= " + CDStatus
                + " CTSStatus= " + CTSStatus
                + " DSRStatus= " + DSRStatus
                + " DTRStatus= " + DTRStatus);
    }

    /** this method uses the modem methods in order to call a phone number and keep the line bussy */
    public void dialToPhone(String phoneNumber) {
        if (openPort()) {
            setLocalEchoOn();
            // set local echo on
            receive("OK"); // expect OK as a response
            resetModem();
            receive("OK"); // expect OK as a response
            sendToPort("ATDT" + phoneNumber); // dial number
            receive("RINGING"); // expect RINGING as a response
        }
    }

    private void resetModem() {
        sendToPort("ATZ"); // reset modem
    }

    private void setLocalEchoOn() {
        sendToPort("ATE1");
    }

    /** this method will send the modem the hang up command and will close the port */
    public void hangUp() {
        sendToPort("ATH0");
        // hang up modem call
        // (and allow hand set to be used without interference)
        receive("OK");
        // expect OK as a response
        closePort();
    }

    /** this is a main test method */
    public static void main(String[] args) {
        SetPath.loadNativeLib();
        try {
            ModemPort mp = new ModemPort("/dev/cu.modem","atdt8770890");
            mp.dialToPhone("877-0890");
            System.out.println("the call is made please pickup the handset and press enter");
            System.in.read();
            mp.hangUp();
        } catch (IOException err) {
            err.printStackTrace();
        }
    }

    public int getBaud() {
        return baud;
    }

    public void setBaud(int baud) {
        this.baud = baud;
    }

    public String getMessageString() {
        return messageString;
    }

    public void setMessageString(String messageString) {
        this.messageString = messageString;
    }

    public String getPortName() {
        return portName;
    }

    public void setPortName(String portName) {
        this.portName = portName;
    }

    public String getApplicationName() {
        return applicationName;
    }

    public void setApplicationName(String myApplicationName) {
        this.applicationName = myApplicationName;
    }

    public int getTimeOutInSeconds() {
        return timeOutInSeconds;
    }

    public void setTimeOutInSeconds(int timeOut) {
        this.timeOutInSeconds = timeOut;
    }
}