package javagroup.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.StringTokenizer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * This is a basic ClassLoader for loading files from directories on the
 * local filesystem either in directories or zipfiles.
 *
 * @author Luke Gorrie
 */
public class StandardLoader extends ClassLoader {

    /**
     * Classpath to search for class files.
     */
    protected Object[] _classPath;

    /**
     * Constructs a new StandardLoader with the specified classpath.  The
     * classpath can contain directories and/or zip-files.
     *
     * @param classPath The classpath to use for this loader.
     */
    public StandardLoader(String classPath) {

        super();

        // each element of the classpath is delimited by the current operating system's path separator
        StringTokenizer st = new StringTokenizer(classPath,
                System.getProperty("path.separator"));

        // store the individual elements in an array to speed things up during loading
        _classPath = new Object[st.countTokens()];
        int index = 0;
        while (st.hasMoreTokens())
            _classPath[index++] = initPath(st.nextToken());
    }

    /**
     * Initializes a path entry.  Directories and zip files are supported.
     *
     * @param name Path name of the entry.
     * @return An Object representing a classpath entry.
     */
    protected Object initPath(String name) {
        // check if this entry is a zip-file
        if (name.substring(name.length() - ".zip".length())
                .equalsIgnoreCase(".zip"))
            return initZipFile(name);
        // if not, assume it is a directory
        else
            return initDirectory(name);
    }

    /**
     * Initialize a directory accessible on the local filesystem.
     *
     * @param name Name of the directory.
     * @return A File representing the directory.
     */
    protected File initDirectory(String name) {
        try {
            // open directory
            File dir = new File(name);

            // ensure a directory was opened.
            if (!dir.isDirectory())
                throw new IOException();

            // return the directory
            return dir;
        } catch (IOException ioe) {
            // return null, directory did not exist.
            return null;
        }
    }

    /**
     * Initialize a zip-file accessible on the local filesystem.
     *
     * @param name Name of the Zip-file
     * @return A ZipFile representing the Zip classpath entry.
     */
    public ZipFile initZipFile(String name) {

        try {
            // open and return the requested zipfile
            ZipFile zip = new ZipFile(name);
            return zip;
        } catch (IOException ioe) {
            // null if the file could not be found
            return null;
        }

    }

    /**
     * Load a class from from available locations.
     */
    public Class loadClass(String name, boolean resolve)
            throws ClassNotFoundException {

        Class clazz = null;

        // check if the class can be found in the system classpath
        try {
            clazz = findSystemClass(name);
        } catch (ClassNotFoundException cnfe) {
        }
        // if not, check if it is preloaded by this classloader.
        if (clazz == null) {
            clazz = findLoadedClass(name);
        }
        // if the class is still not found, try to custom-load it
        if (clazz == null) {

            // array to store class data in
            byte[] class_data = null;

            // try each classpath entry
            for (int i = 0; i < _classPath.length; i++) {

                // if this entry failed to initialize, skip it
                if (_classPath[i] != null) {
                    try {
                        // custom-load for this classpath entry
                        class_data = loadClassDef(_classPath[i], name);
                    } catch (IOException e) {
                    }
                    // if the load was successful, break out
                    if (class_data != null)
                        break;
                }
            }
            // if the class data was loaded, define the class
            if (class_data != null) {
                clazz = defineClass(class_data, 0, class_data.length);
                if (resolve)
                    resolveClass(clazz);

                // success
                return clazz;
            }

        }

        // failure
        throw new ClassNotFoundException("Cannot load class: " + name);
    }

    /**
     * Reads a byte[] from the given path.
     *
     * @param from The path to load from.
     * @param name The name of the class to load.
     * @return A byte[] containing the class data.
     */
    public byte[] loadClassDef(Object from, String name)
            throws IOException {

        // byte array to store class data in
        byte[] data = null;

        // turn the classname into a filename
        name = name.replace('.', File.separatorChar) + ".class";

        // check if this entry is a Directory
        if ((from instanceof File) && ((File) from).isDirectory()) {
            File from_file = (File) from;

            // open file
            File class_file = new File(from_file, name);

            // init array and read class data
            data = new byte[(int) class_file.length()];
            new FileInputStream(class_file).read(data);

            // success
            return data;
        }

        // check if this entry is a zip-file
        if (from instanceof ZipFile) {
            ZipFile zip = (ZipFile) from;

            // find entry in zip for class
            ZipEntry class_entry = zip.getEntry(name);

            // ensure entry exists
            if (class_entry == null)
                throw new IOException("Entry doesn't exist");

            // initalize and read data
            data = new byte[(int) class_entry.getSize()];
            zip.getInputStream(class_entry).read(data);

            // success
            return data;
        }

        // failure
        return null;

    }

}

