/*
 * Copyright (C) 1997, 1998 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.util;

import java.io.*;

/**
 * Standard IO class that holds input/output/error streams.
 *
 * @author Luke Gorrie
 */
public class StandardIO {

    /**
     * true if System.{in, out, err} wrappers are installed
     */
    protected static boolean _systemWrappersInstalled = false;

    // instance streams
    protected PrintStream _out, _err;
    protected InputStream _in;

    // the "system" streams
    protected static PrintStream __out, __err;
    protected static InputStream __in;

    static {
        try {
            ensureSystemWrappersInstalled();
        } catch (RuntimeException e) {
            // don't let any runtime exceptions propagate from the static initializer
        }
    }

    public StandardIO(PrintStream out, InputStream in, PrintStream err) {
        _out = out;
        _err = err;
        _in = in;

        __out = new PrintStream(new StandardOutputStream());
        __in = new StandardInputStream();
        __err = new PrintStream(new StandardErrorStream());
    }

    public static StandardIO getInstance() {
        return (StandardIO) Namespace.getNamespace()
                .getInstanceForClass(StandardIO.class);
    }

    public static synchronized void ensureSystemWrappersInstalled() {
        if (!_systemWrappersInstalled) {
            installSystemWrappers();
        }
        // make sure the System standard-IO streams are the redirectors
        try {
            System.setOut(__out);
            System.setIn(__in);
            System.setErr(__err);
        } catch (SecurityException e) {
            // ignore any security exception
        }
    }

    public static synchronized void installSystemWrappers() {

        if (!_systemWrappersInstalled) {
            _systemWrappersInstalled = true;
            PrintStream out = new PrintStream(
                    new FileOutputStream(FileDescriptor.out), true);
            PrintStream err = new PrintStream(
                    new FileOutputStream(FileDescriptor.out), true);
            InputStream in = new BufferedInputStream(
                    new FileInputStream(FileDescriptor.in));
            StandardIO stdio = new StandardIO(out, in, err);
            Namespace.getNamespace().registerDefaultInstanceForClass(
                    StandardIO.class,
                    stdio);

        }
    }

    public PrintStream getOut() {
        return _out;
    }

    public InputStream getIn() {
        return _in;
    }

    public PrintStream getErr() {
        return _err;
    }

}

