/*
 * Copyright (C) 1997 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.util;

import java.util.Vector;

/**
 * Class to represent a reference-counted resource.  The resource is
 * dispose()d when all "hard" references are released.
 *
 * @author Luke Gorrie
 */
public class Resource implements Disposable {

    // Vector used to reference resources to prevent garbage collection
    private static Vector __resources;

    /**
     * The number of registered references to this Resource.
     */
    protected int _references;

    /**
     * True if this resource is not to be disposed of.
     */
    protected boolean _static;

    protected Vector _disposalListeners;

    protected Disposable _wrappedResource;

    protected boolean _disposed = false;

    public Resource() {
        _disposalListeners = new Vector();
    }

    public Resource(Disposable disposable) {
        this();
        _wrappedResource = disposable;
    }

    /**
     * Lock add a lock. *
     */
    public synchronized void addLock() {
        _references++;
    }

    /**
     * Release a lock. *
     */
    public synchronized void releaseLock() {
        if (!_static) {
            _references--;

            if (_references <= 0)
                this.dispose();
        }
    }

    /**
     * Mark resource as 'static', so it will never be destroyed. *
     */
    public synchronized void setStatic() {
        if (!_static)
            __resources.addElement(this);
        _static = true;
    }

    /**
     * Release all locks and free resource. *
     */
    public void unregister() {

        if (__resources.contains(this))
            __resources.removeElement(this);
    }

    public void dispose() {
        if (_disposed)
            return;

        if (_wrappedResource != null) {
            _wrappedResource.dispose();
        }
        fireResourceDisposalEvent();
        _disposalListeners.removeAllElements();
        _disposed = true;
    }

    public boolean isDisposed() {
        return _disposed;
    }

    public void addDisposalListener(ResourceDisposalListener listener) {
        _disposalListeners.addElement(listener);
    }

    public void removeDisposalListener(ResourceDisposalListener listener) {
        _disposalListeners.removeElement(listener);
    }

    public void fireResourceDisposalEvent() {
        synchronized (_disposalListeners) {
            for (int i = 0; i < _disposalListeners.size(); i++) {
                ResourceDisposalListener listener = (ResourceDisposalListener)
                        _disposalListeners.elementAt(i);
                listener.resourceDisposed(this);
            }
        }
    }

}

