/*
 * This class is free software, do with it as you please.
 */

package javagroup.util;

import java.util.Hashtable;

/**
 * Lightweight pseudo-namespace management class.	Designed as a replacement
 * for static code with state.
 * <p/>
 * TODO: more verbose explanation, discussion of usefulness and
 * implementation pattern essential.
 *
 * @author Luke Gorrie
 */
public class Namespace {

    /**
     * child namespace
     */
    protected Namespace _child;

    // hashtable of instances, keyed by class
    private Hashtable _classToInstanceMap;

    /* static
     */

    // top-level namespace
    private static Namespace __defaultNamespace;

    static {
        __defaultNamespace = new Namespace();
    }

    /**
     * Constructs a Namespace with no root node.	 The root node is assigned
     * when the Namespace is registered.
     */
    public Namespace() {

        _classToInstanceMap = new Hashtable();

    }

    /**
     * Accessor method for gettting the namespace for the calling code.
     * This method should always be used to get a Namespace instance for
     * querying.
     *
     * @return The Namespace object the caller-code is supposed to use.
     */
    public static Namespace getNamespace() {

        return __defaultNamespace;

    }

    /**
     * Register the default instance for a class.	This should only be done
     * from the class that is being registered, and only from its static
     * init {} block, eg: <p> <code> class Foo { static {
     * Namespace.getNamespace() .registerDefaultInstanceForClass(Foo.class,
     * new Foo()); } } </code>
     */
    public static void registerDefaultInstanceForClass(Class klass,
                                                       Object instance) {

        __defaultNamespace.forceRegisterInstanceForClass(klass, instance);

    }

    /**
     * Register a new Namespace instance.
     *
     * @param namespace The new namespace to use.
     */
    public synchronized void registerNamespace(Namespace namespace) {

        if (_child == null) {
            _child = namespace;
        } else
            _child.registerNamespace(namespace);

    }

    /**
     * Register an instance for a class.
     *
     * @param klass    The class to register the instance for.
     * @param instance The instance of that class (must be instanceof the
     *                 class).
     */
    public void registerInstanceForClass(Class klass, Object instance) {

        if (_child != null) {
            _child.registerInstanceForClass(klass, instance);
            return;
        }

        // check that it is an instance of the right class
        if (!klass.isInstance(instance))
            throw new IllegalArgumentException(instance.toString() +
                    " is not an instance of " +
                    klass.getName());

        _classToInstanceMap.put(klass, instance);

    }

    /**
     * Get the registered instance for a given class.
     *
     * @param klass The class to get an instance of.
     * @return The registered instance for the given class.
     */
    public Object getInstanceForClass(Class klass) {

        Object instance;

        if (_child != null) {
            instance = _child.getInstanceForClass(klass);
            if (instance != null) {
                return instance;
            }
        }

        return _classToInstanceMap.get(klass);

    }

    // register instance in this object without consulting the rest of the chain
    void forceRegisterInstanceForClass(Class klass, Object instance) {

        _classToInstanceMap.put(klass, instance);

    }

}


