/*
 * Copyright (C) 1997, 1988 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.*;

import java.net.URL;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * An implementation of the ProcessManager interface. Further documentation
 * to come, please refer to ProcessManager for usage information.
 *
 * @author Luke Gorrie
 * @version $Id: StandardProcessManager.java,v 1.4 1999/01/05 11:46:07 luke
 *          Exp $
 */
public class StandardProcessManager
        implements ProcessManager, ResourceDisposalListener {

    /**
     * default classpath. *
     */
    protected URL[] _classpath;

    /**
     * Hashtable of all processes. *
     */
    protected Hashtable _processes;

    /**
     * SecurityManager *
     */
    protected ProcessSecurityManager _securityManager;

    /**
     * Vector of registered ProcessEventListeners *
     */
    protected Vector _processEventListeners;

    /**
     * Garbage collector *
     */
    protected ProcessGarbageCollector _garbageCollector;

    /**
     * Per-process namespace manager *
     */
    protected ProcessNamespace _namespace;

    /**
     * Queue of garbage processes for the gc to remove. *
     */
    protected Vector _garbageQueue;

    protected ThreadGroup _rootProcessGroup;

    // next pid number
    private long _nextpid;

    /**
     * Constructs a ProcessManager with no default classpath.	This should
     * not be invoked directly, instances are obtained via getInstance().
     */
    public StandardProcessManager() {

        _rootProcessGroup = new ThreadGroup("ThreadGroup for JProcesses");

        // initialize containers
        _processes = new Hashtable();
        _processEventListeners = new Vector();

        _garbageQueue = new Vector();

        // spawn a garbage collector
        _garbageCollector = new ProcessGarbageCollector(this);

        // create and install a namespace for per-process resources
        _namespace = new ProcessNamespace(this);
        Namespace.getNamespace().registerNamespace(_namespace);

        // install security manager
        _securityManager = new ProcessSecurityManager(this);
        System.setSecurityManager(_securityManager);

    }

    /**
     * Constructs a ProcessManager with a given default classpath.	 This
     * should not be invoked directly, instances are obtained via
     * getInstance().
     */
    public StandardProcessManager(URL[] classpath) {
        this();

        _classpath = classpath;

    }

    /**
     * Return the next pid-number and increment the pid counter.
     *
     * @return A unique process-id.
     */
    protected synchronized long getNextPid() {
        return _nextpid++;
    }

    /**
     * Create a process.
     *
     * @param className The name of the target class.
     * @return A process for the target.
     */
    public JProcess createProcess(String className)
            throws ProcessCreationException {
        return createProcess(className, new String[0], null);
    }

    /**
     * Create a process with given arguments.
     *
     * @param className The name of the target class.
     * @param args      The arguments to pass to the main(String[])
     *                  method.
     * @return A process for the target.
     */
    public JProcess createProcess(String className, String[] args)
            throws ProcessCreationException {
        return createProcess(className, args, null);
    }

    /**
     * Create a process with given args, and additional classpath(s).
     *
     * @param className The name of the target class.
     * @param args      The arguments to pass to the main(String[])
     *                  method.
     * @param classpath An array of URLs to search for classes in.
     */
    public synchronized JProcess createProcess(String className, String[] args,
                                               URL[] classpath)
            throws ProcessCreationException {

        // ensure standard io redirectors are in place
        StandardIO.ensureSystemWrappersInstalled();

        if (classpath == null) {
            classpath = _classpath;
        } else {
            classpath = mergeWithDefaultClasspath(classpath);
        }

        // instantiate process
        StandardJProcess process =
                new StandardJProcess(className,
                        getNextPid(),
                        args,
                        _rootProcessGroup,
                        classpath);

        // create a token resource and bind it to the process.
        // when the resource is released, it will indicate that the process
        // is dead and it can be garbage collected.
        ProcessLifeToken token = new ProcessLifeToken(process);
        token.addDisposalListener(this);
        process.registerResource(token);

        // inform listeners that a process has been created
        fireProcessCreationEvent(process);

        // register process in process table
        _processes.put(new Long(process.getPid()), process);

        // create an empty namespace for the process
        _namespace.registerNamespaceForProcess(new Namespace(), process);

        return process;

    }

    /**
     * Creates a process from the information in a given String of the
     * format: "&lt;classname&gt; &lt;arg&gt;*".
     *
     * @param string The string to decode process info from.
     * @return The created JProcess.
     */
    public JProcess createProcessFromString(String string)
            throws ProcessCreationException {

        return createProcessFromString(string, null);

    }

    /**
     * Creates a process from the information in a given String of the
     * format: "[-eclasspath url[,url]*] <classname> <arg>*".
     *
     * @param string    The string to decode process info from.
     * @param classpath Classpaths to search in addition to defaults.
     * @return The created JProcess.
     */
    public JProcess createProcessFromString(String string,
                                            URL[] classpath)
            throws ProcessCreationException {

        // check that a classname can be extracted
        StringTokenizer tok = new StringTokenizer(string, " ");
        if (tok.countTokens() < 1)
            throw new ProcessCreationException(
                    "Invalid parameter string.");

        // class name
        String class_name = tok.nextToken();

        if (class_name.equals("-eclasspath")) {
            if (!tok.hasMoreTokens()) {
                throw new ProcessCreationException(
                        "Must specify a classpath with -eclasspath option");
            }
            URL[] arg_classpath = URLClassLoader.decodePathString(
                    tok.nextToken());
            if (classpath == null || classpath.length == 0) {
                classpath = arg_classpath;
            } else if (arg_classpath != null || arg_classpath.length != 0) {
                int combined_length = classpath.length +
                        arg_classpath.length;
                URL[] combined_classpath = new URL[combined_length];
                System.arraycopy(arg_classpath,
                        0,
                        combined_classpath,
                        0,
                        arg_classpath.length);
                System.arraycopy(classpath, 0, combined_classpath,
                        arg_classpath.length, combined_classpath.length);
                classpath = combined_classpath;
            }
            if (!tok.hasMoreTokens()) {
                throw new ProcessCreationException(
                        "Must specify a classname");
            }
            class_name = tok.nextToken();
        }

        // read args from string
        String[] args = new String[tok.countTokens()];
        for (int i = 0; i < args.length; i++)
            args[i] = tok.nextToken();


        // create and return process
        return createProcess(class_name, args, classpath);

    }

    /**
     * Get the process that a threadgroup belongs to.
     *
     * @param group The ThreadGroup to attribute a process to.
     * @return The process owning the group, or null if none found.
     */
    public JProcess getProcessFor(ThreadGroup group) {

        // enumerate all registered processes
        Enumeration processes = _processes.elements();

        JProcess match = null;

        // cycle until a match is found, or all processes are tested
        while ((match == null) && (processes.hasMoreElements())) {
            JProcess process = (JProcess) processes.nextElement();
            // get threadgroup from process
            ThreadGroup process_group = process.getThreadGroup();
            // if the process' group is an ancestor of the group being checked,
            // make match
            if (process_group.parentOf(group))
                match = process;
        }

        // return matched process, or null if none were found
        return match;

    }

    /**
     * Get the process whose target class was loaded by a given
     * ClassLoader.
     *
     * @param loader The classloader to check for.
     * @return The process owning the loader, or null if none found.
     */
    public JProcess getProcessFor(ClassLoader loader) {

        // enumerate all registered processes
        Enumeration processes = _processes.elements();

        JProcess match = null;

        // cycle until a match if found, or all processes are tested
        while ((match == null) && (processes.hasMoreElements())) {
            JProcess process = (JProcess) processes.nextElement();
            // get classloader from process
            ClassLoader process_loader = process.getClassLoader();
            // if both classloader variables reference the same instance,
            // make a match
            if (loader == process_loader)
                match = process;
        }

        // return mached process, or null if none found
        return match;

    }

    protected URL[] mergeWithDefaultClasspath(URL[] classpath) {

        // check if either is blank
        if (_classpath == null) return classpath;
        if (classpath == null) return _classpath;

        URL[] result = new URL[classpath.length + _classpath.length];
        for (int i = 0; i < classpath.length; i++) {
            result[i] = classpath[i];
        }
        for (int i = 0; i < _classpath.length; i++) {
            result[i + classpath.length] = _classpath[i];
        }
        return result;
    }

    /**
     * Return the current process.
     *
     * @return The process making the invocation, or null if none can be
     *         determined.
     */
    public JProcess getCurrentProcess() {
        // get security manager to determine current process and return it
        return _securityManager.getCurrentProcess();
    }

    /**
     * Get a process by process-id.
     *
     * @param pid The id-number of the process.
     * @return The process, or null if no match.
     */
    public JProcess getProcess(long pid) {
        return (JProcess) _processes.get(new Long(pid));
    }

    /**
     * Get an enumeration of all processes.
     *
     * @return Enumeration of all processes.
     */
    public Enumeration getProcesses() {
        return _processes.elements();
    }

    /**
     * Kill a process by pid.
     *
     * @param pid The process-id to kill.
     */
    public synchronized void kill(long pid) {

        // get process by pid

        // KIRR: in my opinion, remove is not needed as it will be invoked by the listener
        JProcess process = (JProcess) _processes.remove(new Long(pid));
        if (process != null) {
            // queue process for garbage collection
            _garbageQueue.addElement(process);
            // alert the garbage collector
            _garbageCollector.wakeUp();
            // no longer killed here.	 the process garbage collector is used to
            // perform the kill because it runs on a safe thread
            //process.kill();
        }
    }

    /* Event stuff - self-explanatory
     */

    public void addProcessEventListener(ProcessEventListener listener) {
        _processEventListeners.addElement(listener);
    }

    public void removeProcessEventListener(ProcessEventListener listener) {
        _processEventListeners.removeElement(listener);
    }

    protected void fireProcessDestructionEvent(JProcess process) {
        synchronized (_processEventListeners) {
            for (int i = 0; i < _processEventListeners.size(); i++) {
                ProcessEventListener listener =
                        (ProcessEventListener) _processEventListeners.elementAt(
                                i);
                listener.processDestroyed(process);
            }
        }
    }

    protected void fireProcessCreationEvent(JProcess process) {
        synchronized (_processEventListeners) {
            for (int i = 0; i < _processEventListeners.size(); i++) {
                ProcessEventListener listener =
                        (ProcessEventListener) _processEventListeners.elementAt(
                                i);
                listener.processCreated(process);
            }
        }
    }

    /**
     * For ResourceDisposalListener interface. *
     */
    public void resourceDisposed(Resource resource) {

        if (resource instanceof ProcessLifeToken) {
            JProcess process = ((ProcessLifeToken) resource).getProcess();
            kill(process.getPid());
        }

    }

    /**
     * Perform garbage collection.	 Usually invoked by the garbage
     * collector.
     */
    public synchronized void doGarbageCollect() {

        // restore the standard-io redirectors if someone's tampered
        // with them
        StandardIO.ensureSystemWrappersInstalled();

        Enumeration garbage = _garbageQueue.elements();
        while (garbage.hasMoreElements()) {

            StandardJProcess process = (StandardJProcess) garbage.nextElement();
            _namespace.removeNamespaceForProcess(process);
            fireProcessDestructionEvent(process);

        }

        _garbageQueue.removeAllElements();

        Enumeration processes = _processes.elements();
        while (processes.hasMoreElements()) {

            StandardJProcess process = (StandardJProcess) processes.nextElement();
            process.tryGarbageCollect();

        }

        System.gc();

    }

    /**
     * Get the Namespace object for the given process.
     *
     * @param process The process to get Namespace for.
     * @return The appropriate namespace.
     */
    public Namespace getNamespaceForProcess(JProcess process) {
        return _namespace.getNamespaceForProcess(process);
    }

    /**
     * Set the streams to be used when the Process asks for standard io.
     *
     * @param process The process to set for.
     * @param stdio   The set of IO streams to use as standard io.
     */
    public void setStandardIOForProcess(JProcess process,
                                        StandardIO stdio) {
        Namespace namespace = getNamespaceForProcess(process);
        namespace.registerInstanceForClass(StandardIO.class, stdio);
    }

    /**
     * Get the streams a process uses for standard io
     *
     * @param process The process to get streams for.
     * @return The set of streams being used as standard io.
     */
    public StandardIO getStandardIOForProcess(JProcess process) {
        Namespace namespace = getNamespaceForProcess(process);
        return (StandardIO) namespace.getInstanceForClass(
                StandardIO.class);
    }

}


