/*
 * Copyright (C) 1997, 1998 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.Resource;
import javagroup.util.ResourceDisposalListener;
import javagroup.util.URLClassLoader;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.Vector;

/**
 * An implementation of the JProcess interface.	Should only be used by a
 * ProcessManager.
 *
 * @author Luke Gorrie
 * @version $Id: StandardJProcess.java,v 1.4 1999/01/05 11:45:07 luke Exp
 *          $
 */
public class StandardJProcess implements Runnable,
                                         JProcess,
                                         ResourceDisposalListener {
    /**
     * Unique process-id
     */
    protected long _pid;
    /**
     * Process name
     */
    protected String _name;
    /**
     * Classloader used to load target class
     */
    protected URLClassLoader _classLoader;
    /**
     * Threadgroup for threads in this process
     */
    protected ThreadGroup _processThreadGroup;
    /**
     * Resources locked by this process. *
     */
    protected Vector _lockedResources;
    /**
     * Resources this process is locked to *
     */
    protected Vector _resourcesLockedTo;
    /**
     * state of process *
     */
    protected int _state;
    /**
     * exit code
     */
    protected int _exitCode;

    // "public static void main(String[])" method of target class
    private Method _targetMethod;
    // args to pass to main method
    private String[] _args;
    // thread used to invoke main method
    private Thread _mainThread;

    /**
     * Creates a process.	This should invoked by a ProcessManager.
     *
     * @param className The fully-qualified name of the target class.
     * @param args      The arguments to pass to the target's
     *                  main(String[]) method.
     * @param pid       The process-id number.
     * @param classpath A list of URLs to check for classes.
     */
    public StandardJProcess(String className, long pid, String[] args,
                            ThreadGroup parent, URL[] classpath)
            throws ProcessCreationException {

        _name = className;
        _args = args;

        setState(UNSTARTED);

        _lockedResources = new Vector();
        _resourcesLockedTo = new Vector();

        if (_args == null)
            _args = new String[0];

        // append the arguments to the process' name to make it more descriptive
        for (int i = 0; i < _args.length; i++)
            _name += " " + _args[i];

        _pid = pid;

        // class factory method to create a classloader
        _classLoader = createClassLoader();

        // add classpath
        if (classpath != null)
            _classLoader.addClassPath(classpath);

        // create a threadgroup, to be used exclusively by this process
        _processThreadGroup = createThreadGroup(parent);

        // get a Method object for 'public static void main(String[])'
        try {
            setTargetClass(className);
        } catch (Exception e) {
            throw new ProcessCreationException(e.toString());
        }

    }

    /**
     * Start the process running. *
     */
    public void launch() {
        // a 'main' thread for the process
        _mainThread = new Thread(_processThreadGroup, this,
                "Main thread for " + _name);
        // start process
        _mainThread.start();
        setState(RUNNING);
        Thread.yield();

    }

    public void addClassPath(URL[] classpath) {
        _classLoader.addClassPath(classpath);
    }

    /**
     * Should not be called directly. *
     */
    public void run() {
        try {
            // create argument array
            Object[] args_array = {_args};
            // invoke target method, starting the target application
            _targetMethod.invoke(null, args_array);
        } catch (InvocationTargetException e) {
            //System.out.println("Invocation target exception: "+e);
            //e.getTargetException().printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
            //System.out.println(e);
            //throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * Factory method for creating a classloader.
     *
     * @return A URLClassLoader for loading the target class.
     */
    protected URLClassLoader createClassLoader() {
        return new URLClassLoader();
    }

    /**
     * Factory method for creating a process ThreadGroup.
     *
     * @return A ThreadGroup for threads in the process.
     */
    protected ThreadGroup createThreadGroup(ThreadGroup parent) {
        ThreadGroup group = new ThreadGroup(parent, this.toString());
        group.setMaxPriority(Thread.MAX_PRIORITY - 1);

        return group;
    }

    // get main method from target class
    private void setTargetClass(String className)
            throws ClassNotFoundException, NoSuchMethodException {

        // load target class
        Class target_class = _classLoader.loadMainClass(className);
        // argument list ( String[] )
        Class[] arg_types = {String[].class};
        // get target method
        _targetMethod = target_class.getMethod("main", arg_types);
        // check that the method is static
        if (((_targetMethod.getModifiers() & Modifier.STATIC) == 0)
                || (_targetMethod.getModifiers() & Modifier.PUBLIC) == 0)
            throw new NoSuchMethodException("main(String[]) method of " +
                    target_class.getName() +
                    " is not public and static.");

    }

    /**
     * Return the classloader used to load the target class
     *
     * @return The ClassLoader for this process.
     */
    public ClassLoader getClassLoader() {
        return _classLoader;
    }

    /**
     * Return the ThreadGroup for threads in this process.
     *
     * @return The ThreadGroup for this process.
     */
    public ThreadGroup getThreadGroup() {
        return _processThreadGroup;
    }

    /**
     * Register a resource.
     *
     * @param resource The resource to be registered/locked.
     */
    public synchronized void registerResource(Resource resource) {
        if (!_lockedResources.contains(resource)) {
            _lockedResources.addElement(resource);
            resource.addLock();
        }
    }

    /**
     * Register and bind to a resource.
     *
     * @param resource The resource to lock and bind to.
     */
    public synchronized void registerAndBindToResource(Resource resource) {
        registerResource(resource);
        bindToResource(resource);
    }

    public synchronized void bindToResource(Resource resource) {
        if (!_resourcesLockedTo.contains(resource)) {
            _resourcesLockedTo.addElement(resource);
            resource.addDisposalListener(this);
        }
    }

    /**
     * Release all resource locks. *
     */
    public synchronized void releaseResources() {
        Resource[] resources = new Resource[_lockedResources.size()];
        _lockedResources.copyInto(resources);
        for (int i = 0; i < resources.length; i++) {
            releaseResource(resources[i]);
            //resource.releaseLock();
        }
    }

    protected synchronized void releaseResource(Resource resource) {
        if (_lockedResources.contains(resource)) {
            try {
                resource.releaseLock();
            } catch (Exception e) {
            }
            try {
                _lockedResources.removeElement(resource);
            } catch (Exception e) {
            }
        }

        if (_resourcesLockedTo.contains(resource))
            _resourcesLockedTo.removeElement(resource);

    }

    public void resourceDisposed(Resource resource) {
        releaseResource(resource);
    }



    /**
     * This is the second stage in killing a process. Just doing stop on
     * its threadgroup does not have an immediate effect. For this reason,
     * the threads of the process are destroyed after a while, when the
     * garbage collector wakes up.
     */
    private synchronized void kill_destroyThreadGroup() {
        if (_processThreadGroup != null) {
            _processThreadGroup.destroy();
            _processThreadGroup = null;
        }
    }

    /**
     * Checks if the process is finished running, and if so kills it off to
     * clean up the process table and allow the ClassLoader etc to be
     * garbage collected.
     */
    public synchronized void tryGarbageCollect() {
        // if ths process has no threads, and is not locked to any resources,
        // kill (garbage collect) it.

        if (_state != UNSTARTED && _state != DEAD) {

            if (_resourcesLockedTo.isEmpty() &&
                    _processThreadGroup.activeCount() <= 0) {
                kill_destroyThreadGroup();
            }
        }
    }

    /**
     * Return the process-id.
     *
     * @return Process-id for this process.
     */
    public long getPid() {
        return _pid;
    }

    /**
     * Get the name of this process.
     *
     * @return A reasonably meaningful name.
     */
    public String getName() {
        return _name;
    }

    public int getState() {
        return _state;
    }

    protected synchronized void setState(int state) {
        _state = state;
        notifyAll();
    }

    /**
     * Wait for the process to finish.
     */
    public synchronized void waitFor() {
        while (getState() != DEAD) {
            try {
                wait();
            } catch (InterruptedException e) {
            }
        }
    }

    public int getExitCode() {
        return _exitCode;
    }

    public void setExitCode(int code) {
        _exitCode = code;
    }

    public String toString() {
        return String.valueOf(_pid) + ": " + _name;
    }

}

