/*
 * Copyright (C) 1997 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.Namespace;

import java.util.Hashtable;

/**
 * Namespace used for mapping per-process namespaces.
 *
 * @author Luke Gorrie
 */
public class ProcessNamespace extends Namespace {

    /**
     * Reference to the ProcessManager the object is managing a namespace
     * for.
     */
    protected ProcessManager _processManager;

    /**
     * Hashtable mapping processes to Namespaces *
     */
    protected Hashtable _processToNamespaceMap;

    public ProcessNamespace(ProcessManager processManager) {
        _processManager = processManager;
        _processToNamespaceMap = new Hashtable();
    }

    public synchronized void registerNamespaceToProcess(
            Namespace namespace,
            JProcess process) {

        // choose namespace for process
        if (process != null) {
            Namespace process_namespace = getNamespaceForProcess(process);
            if (process_namespace != null) {
                process_namespace.registerNamespace(namespace);
            }
        }

        // if no process, use default namespace
        else
            super.registerNamespace(namespace);

    }

    public synchronized void registerNamespace(Namespace namespace) {
        registerNamespaceToProcess(namespace,
                _processManager.getCurrentProcess());
    }

    public synchronized void registerNamespaceForProcess(
            Namespace namespace,
            JProcess process) {
        _processToNamespaceMap.put(process, namespace);
    }

    public synchronized Namespace getNamespaceForProcess(JProcess process) {

        if (process == null)
            return null;

        // get namespace from map
        Namespace process_namespace =
                (Namespace) _processToNamespaceMap.get(process);

        if (process_namespace == null) {
            //try { Thread.sleep(100); } catch (Throwable t) {}
            process_namespace = new Namespace();
            _processToNamespaceMap.put(process, process_namespace);
            //_processToNamespaceMap.put(System.out, process_namespace);
        }

        // null if no match
        return process_namespace;
    }

    public synchronized void removeNamespaceForProcess(JProcess process) {

        _processToNamespaceMap.remove(process);

    }

    public synchronized
    void registerInstanceForClass(Class klass, Object instance) {
        Namespace namespace =
                getNamespaceForProcess(
                        _processManager.getCurrentProcess());

        if (namespace != null)
            namespace.registerInstanceForClass(klass, instance);

        else
            super.registerInstanceForClass(klass, instance);

    }

    public synchronized Object getInstanceForClass(Class klass) {

        Namespace namespace =
                getNamespaceForProcess(
                        _processManager.getCurrentProcess());

        Object instance = null;

        if (namespace != null)
            instance = namespace.getInstanceForClass(klass);

        if (instance == null)
            instance = super.getInstanceForClass(klass);

        return instance;

    }

}

