/*
 * Copyright (C) 1997 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.Namespace;
import javagroup.util.URLClassLoader;

import java.net.URL;

/**
 * Place holder for accessing a ProcessManager.
 *
 * @author Luke Gorrie
 */
public class ProcessManagerHolder {

    /**
     * Return a ProcessManager instance.  This method should be used to
     * access ProcessManagers rather than direct instantiation.
     */
    public static synchronized ProcessManager getProcessManager() {

        // grab ProcessManager instance
        ProcessManager manager = (ProcessManager)
                Namespace.getNamespace().getInstanceForClass(
                        ProcessManager.class);
        // if there isn't one, create one using the default implementation
        if (manager == null) {
            manager = instantiateProcessManager();

            // register the default manager for the Namespace
            Namespace
                    .registerDefaultInstanceForClass(ProcessManager.class,
                            manager);
        }

        return manager;
    }

    private static ProcessManager instantiateProcessManager() {

        // default classpath
        URL[] paths =
                URLClassLoader.decodePathString(
                        System.getProperty("default.classpath"));

        ProcessManager manager;

        // if possible, instantiate a process manager with the default classpath
        if (paths != null)
            manager = new StandardProcessManager(paths);
        else
            manager = new StandardProcessManager();

        return manager;
    }

}

