/*
 * Copyright (C) 1997 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.StandardIO;

import java.net.URL;
import java.util.Enumeration;

public interface ProcessManager {

    /**
     * Create a process with given args, and additional classpath(s).
     *
     * @param className The name of the target class.
     */
    public JProcess createProcess(String className)
            throws ProcessCreationException;

    /**
     * Create a process with given args, and additional classpath(s).
     *
     * @param className The name of the target class.
     * @param args      The arguments to pass to the main(String[])
     *                  method.
     */
    public JProcess createProcess(String className, String[] args)
            throws ProcessCreationException;

    /**
     * Create a process with given args, and additional classpath(s).
     *
     * @param className The name of the target class.
     * @param args      The arguments to pass to the main(String[])
     *                  method.
     * @param classPath An array of URLs to search for classes in.
     */
    public JProcess createProcess(String className, String[] args,
                                  URL[] classPath)
            throws ProcessCreationException;

    /**
     * Creates a process from the information in a given String of the
     * format: "<classname> <arg>*".
     *
     * @param line The string to decode process info from.
     * @return The created JProcess.
     */
    public JProcess createProcessFromString(String line)
            throws ProcessCreationException;

    /**
     * Creates a process from the information in a given String of the
     * format: "<classname> <arg>*".
     *
     * @param line    The string to decode process info from.
     * @param classPath Classpaths to search in addition to defaults.
     * @return The created JProcess.
     */
    public JProcess createProcessFromString(String line, URL[] classPath)
            throws ProcessCreationException;


    /**
     * Get the process that a threadgroup belongs to.
     *
     * @param group The ThreadGroup to attribute a process to.
     * @return The process owning the group, or null if none found.
     */
    public JProcess getProcessFor(ThreadGroup group);

    /**
     * Get the process that a classloader belongs to.
     *
     * @return The process owning the group, or null if none found.
     */
    public JProcess getProcessFor(ClassLoader loader);

    /**
     * Return the current process.
     *
     * @return The process making the invocation, or null if none can be
     *         determined.
     */
    public JProcess getCurrentProcess();

    /**
     * Get a process by process-id.
     *
     * @param pid The id-number of the process.
     * @return The process, or null if no match.
     */
    public JProcess getProcess(long pid);


    /**
     * Get an enumeration of all processes.
     *
     * @return Enumeration of all processes.
     */
    public Enumeration getProcesses();

    /**
     * Kill a process by pid.
     *
     * @param pid The process-id to kill.
     */
    public void kill(long pid);

    public void addProcessEventListener(ProcessEventListener listener);

    public void removeProcessEventListener(ProcessEventListener listener);

    /**
     * Perform garbage collection.  Usually invoked by the garbage
     * collector.
     */
    public void doGarbageCollect();

    /**
     * Set the streams to be used when the Process asks for standard io.
     *
     * @param process The process to set for.
     */
    public StandardIO getStandardIOForProcess(JProcess process);

    /**
     * Get the streams a process uses for standard io
     *
     * @param process The process to get streams for.
     * @param stdio   The set of IO streams to use as standard io.
     */
    public void setStandardIOForProcess(JProcess process,
                                        StandardIO stdio);

}
