/*
 * Copyright (C) 1997 Luke Gorrie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 */

package javagroup.process;

import javagroup.util.Resource;

/**
 * An interface for classes representing processes inside the JVM.
 *
 * @author Luke Gorrie
 */
public interface JProcess {

    /**
     * State constant.
     */
    public static final int UNSTARTED = 0;
    /**
     * State constant.
     */
    public static final int RUNNING = 1;
    /**
     * State constant.
     */
    public static final int DEAD = 2;

    /**
     * Gets the name of this process.
     *
     * @return The process name.
     */
    public String getName();

    /**
     * Gets the unique ID number for the process.
     *
     * @return JProcess ID.
     */
    public long getPid();

    /**
     * Set the process running on its own Thread.
     */
    public void launch();

    /**
     * Get the state of the process.
     *
     * @return the state, matching one of the state constants
     */
    public int getState();

    /**
     * Return the ThreadGroup for this JProcess.  All Threads belonging to
     * the process must be members of this ThreadGroup, or of its children
     *
     * @return The process ThreadGroup.
     */
    public ThreadGroup getThreadGroup();

    /**
     * Return the ClassLoader that was used to load the classes for this
     * process.
     *
     * @return The ClassLoader used, or null if not appropriate.
     */
    public ClassLoader getClassLoader();

    /**
     * Binds a given resource the the process.  The process will lock the
     * resource, and release the lock upon process death.
     *
     * @param resource The resource to bind to the process.
     */
    public void registerResource(Resource resource);

    /**
     * Binds the process to a resource.  The process will lock the
     * resource, but the process will not be garbage collected while the
     * resource is valid, or it is explicitly killed.  When the process
     * dies, it will release it's lock on the resource.
     *
     * @param resource The resource to bind the process to.
     */
    public void registerAndBindToResource(Resource resource);

    /**
     * Wait until the process terminates.
     */
    public void waitFor();

    /**
     * Get the exit code of the process. This method blocks until the
     * process is terminated. <br> <i>Not implemented</i>
     */
    public int getExitCode();

}

