package javagroup.misc;

import javagroup.disposer.SocketDisposer;
import javagroup.process.JProcess;
import javagroup.process.ProcessManager;
import javagroup.process.ProcessManagerHolder;
import javagroup.util.NullInputStream;
import javagroup.util.Resource;
import javagroup.util.StandardIO;

import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;

/**
 * A service to listen on a socket and launch processes from client
 * requests. Written as a basic example use of the multiprocess api, this
 * code has been kept simple.  A couple of features omitted are a
 * trusted-host list, and thread-per-client communication to allow
 * asynchronous requests.
 *
 * @author Luke Gorrie
 * @version $Revision: 1.1 $
 */
public class ProcessService implements Runnable {

    /**
     * Default port to listen on. *
     */
    public static final int DEFAULT_PORT = 7080;

    /**
     * ServerSocket to take connections from. *
     */
    protected ServerSocket _serverSocket;

    public ProcessService() throws IOException {
        this(DEFAULT_PORT);
    }

    public ProcessService(int port) throws IOException {
        // bind a ServerSocket to take connections on
        // raises IOException
        _serverSocket = new ServerSocket(port);

        new Thread(this).start();
    }

    /**
     * Drives a thread to serve clients by creating processes. *
     */
    public void run() {
        try {
            // loop until thread is explicitly killed
            while (true) {

                // socket connection
                Socket client_sock = null;

                // PrintStream for writing to socket
                PrintStream output = null;

                try {
                    // accept a connection from a client
                    // raises IOException
                    client_sock = _serverSocket.accept();

                    // streams for process standard io
                    PrintStream out, err;
                    InputStream in;

                    // standard out and err are redirected to the socket
                    // raises IOException
                    out =
                            err =
                            output =
                            new PrintStream(client_sock.getOutputStream());

                    // use an empty stream for standard in
                    in = new NullInputStream();

                    // create standard-io set for process
                    StandardIO stdio = new StandardIO(out, in, err);

                    // create a stream to read command from
                    // raises IOException
                    DataInputStream input = new DataInputStream(
                            client_sock.getInputStream());

                    // read the command from the client
                    // raises IOException
                    BufferedReader br = new BufferedReader(
                            new InputStreamReader(input));
                    String command =
                            br.readLine();

                    // get a ProcessManager to launch the process
                    ProcessManager process_manager = ProcessManagerHolder.getProcessManager();

                    // create process
                    // raises ProcessCreationException
                    JProcess process = process_manager.createProcessFromString(
                            command);

                    // set standard io streams for process
                    process_manager.setStandardIOForProcess(process,
                            stdio);

                    // create Resource to bind the socket to the process
                    Resource socket_resource = new Resource(
                            new SocketDisposer(client_sock));

                    // register the resource to the process so that it is disposed
                    // when the process dies
                    process.registerResource(socket_resource);

                    // launch process
                    process.launch();
                } catch (Exception e) {
                    try {
                        output.println(e);
                        client_sock.close();
                    } catch (Exception ioe) {
                    }
                }
            }
        } catch (ExceptionInInitializerError e) {
            e.getException().printStackTrace();
        }
    }

    /**
     * Closes the ServerSocket which was bound to listen for client
     * requests.
     */
    public void finalize() {
        try {
            _serverSocket.close();
        } catch (IOException e) {
        }
    }

    /**
     * Starts a ProcessService on the default port.
     *
     * @param args Ignored.
     */
    public static void main(String[] args) throws Exception {
        new ProcessService();
    }

}

	
