package j3d.cr325;

import gui.run.RunJob;

import javax.media.j3d.*;
import javax.vecmath.Vector3d;
import java.awt.*;
import java.awt.event.KeyEvent;
import java.util.Enumeration;


/**
 * KeyBehavior is a generic behavior class to take key presses and move a
 * TransformGroup through a Java3D scene. The actions resulting from the key strokes
 * are modified by using the Ctrl, Alt and Shift keys.
 * <p/>
 * (version 1.0) reconstructed class to make more generic.
 * <p/>
 * MODIFIED:
 *
 * @author Andrew AJ Cain, Swinburne University, Australia
 *         <acain@it.swin.edu.au>
 *         <p/>
 *         edited from code by:
 *         Gary S. Moss <moss@arl.mil>
 *         U. S. Army Research Laboratory
 *         <p/>
 *         CLASS NAME:
 *         KeyBehavior
 *         <p/>
 *         PUBLIC FEATURES:
 *         // Data
 *         <p/>
 *         // Constructors
 *         <p/>
 *         // Methods:
 *         <p/>
 *         COLLABORATORS:
 * @version 1.0, 25 September 1998 aajc
 */
public class KeyBehavior extends Behavior {
    protected static final double FAST_SPEED = 2.0;
    protected static final double NORMAL_SPEED = 1.0;
    protected static final double SLOW_SPEED = 0.1;

    private TransformGroup viewTransformGroup;
    private Transform3D viewTransform3D;
    private WakeupCondition keyCriterion;

    private final static double TWO_PI = (2.0 * Math.PI);
    private double eps = 32;
    private double rotateXAmount = Math.PI / eps;
    private double rotateYAmount = Math.PI / eps;
    private double rotateZAmount = Math.PI / eps;

    private double moveRate = 0.1;
    private double speed = NORMAL_SPEED;

    private int forwardKey = KeyEvent.VK_UP;
    private int backKey = KeyEvent.VK_DOWN;
    private int leftKey = KeyEvent.VK_LEFT;
    private int rightKey = KeyEvent.VK_RIGHT;
    private int spaceKey = KeyEvent.VK_SPACE;

    TransformGroup gunTransformGroup = null;
    TransformGroup bulletTransformGroup = null;
    TransformGroup targetTransformGroup = null;

    public void setTargetTransformGroup(TransformGroup targetTransformGroup){
        this.targetTransformGroup = targetTransformGroup;
    }

    public void setBulletTransformGroup(TransformGroup bulletTransformGroup) {
        this.bulletTransformGroup = bulletTransformGroup;
    }

    public void setGunTransformGroup(TransformGroup gunTransformGroup) {
        this.gunTransformGroup = gunTransformGroup;
    }

    public KeyBehavior(TransformGroup viewTransformGroup) {
        this.viewTransformGroup = viewTransformGroup;
        viewTransform3D = new Transform3D();

    }

    public void initialize() {
        WakeupCriterion[] keyEvents = new WakeupCriterion[2];

        keyEvents[0] = new WakeupOnAWTEvent(KeyEvent.KEY_PRESSED);
        keyEvents[1] = new WakeupOnAWTEvent(KeyEvent.KEY_RELEASED);
        keyCriterion = new WakeupOr(keyEvents);
        wakeupOn(keyCriterion);
    }


    public void processStimulus(Enumeration criteria) {
        WakeupCriterion wakeup;
        AWTEvent[] event;

        while (criteria.hasMoreElements()) {
            wakeup = (WakeupCriterion) criteria.nextElement();
            if (!(wakeup instanceof WakeupOnAWTEvent))
                continue;

            event = ((WakeupOnAWTEvent) wakeup).getAWTEvent();
            for (int i = 0; i < event.length; i++) {
                if (event[i].getID() == KeyEvent.KEY_PRESSED) {
                    processKeyEvent((KeyEvent) event[i]);
                }
            }
        }
        wakeupOn(keyCriterion);
    }

    protected void processKeyEvent(KeyEvent ke) {

        int keycode = ke.getKeyCode();

        if (ke.isShiftDown())
            speed = FAST_SPEED;
        else
            speed = NORMAL_SPEED;
        if (ke.isAltDown())
            altMove(keycode);
        else if (ke.isControlDown())
            controlMove(keycode);
        else
            standardMove(keycode);
        //moveGun();
    }


    //moves forward backward or rotates left right
    private void standardMove(int kc) {
        System.out.println(kc);
        if (kc == forwardKey)
            moveForward();
        else if (kc == backKey)
            moveBackward();
        else if (kc == leftKey)
            rotLeft();
        else if (kc == rightKey)
            rotRight();
        else if (kc == spaceKey)
            fireBullet();
    }

    //moves left right, rotate up down
    protected void altMove(int keycode) {
        if (keycode == forwardKey)
            rotUp();
        else if (keycode == backKey)
            rotDown();
        else if (keycode == leftKey)
            moveLeft();
        else if (keycode == rightKey)
            moveRight();
    }

    //move up down, rot left right
    protected void controlMove(int keycode) {
        if (keycode == forwardKey)
            moveUp();
        else if (keycode == backKey)
            moveDown();
        else if (keycode == leftKey)
            rollLeft();
        else if (keycode == rightKey)
            rollRight();
    }

    private void moveForward() {
        doMove(new Vector3d(0.0, 0.0, -getMovementRate()));
    }

    private void moveBackward() {
        doMove(new Vector3d(0.0, 0.0, getMovementRate()));
    }

    private void moveLeft() {
        doMove(new Vector3d(-getMovementRate(), 0.0, 0.0));
    }

    private void moveRight() {
        doMove(new Vector3d(getMovementRate(), 0.0, 0.0));
    }

    private void moveUp() {
        doMove(new Vector3d(0.0, getMovementRate(), 0.0));
    }

    private void moveDown() {
        doMove(new Vector3d(0.0, -getMovementRate(), 0.0));
    }

    protected void rotRight() {
        doRotateY(getRotateRightAmount(), getRotateRightAmount());
    }

    protected void rotUp() {
        doRotateX(getRotateUpAmount(), getRotateUpAmount());
    }

    protected void rotLeft() {
        doRotateY(getRotateLeftAmount(), getRotateLeftAmount());
    }

    protected void rotDown() {
        doRotateX(getRotateDownAmount(), getRotateDownAmount());
    }

    protected void rollLeft() {
        doRotateZ(getRollLeftAmount(), getRollLeftAmount());
    }

    protected void rollRight() {
        doRotateZ(getRollRightAmount(), getRollRightAmount());
    }

    protected void doRotateY(double radians, double theMove) {
        viewTransformGroup.getTransform(viewTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotY(radians);
        viewTransform3D.mul(toMove);
        viewTransformGroup.setTransform(viewTransform3D);

        if (gunTransformGroup == null) return;
        Vector3d gunMove = new Vector3d(1, 1, 1);
        Transform3D gunTransform3D = new Transform3D();
        gunTransform3D.rotY(radians * theMove);
        gunTransform3D.setTranslation(gunMove);
        viewTransform3D.mul(gunTransform3D);
        gunTransformGroup.setTransform(viewTransform3D);
    }

    protected void doRotateX(double radians, double theMove) {
        viewTransformGroup.getTransform(viewTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotX(radians);
        viewTransform3D.mul(toMove);
        viewTransformGroup.setTransform(viewTransform3D);

        if (gunTransformGroup == null) return;

        Vector3d gunMove = new Vector3d(1, 1, 1);
        Transform3D gunTransform3D = new Transform3D();
        gunTransform3D.rotX(radians * theMove);
        gunTransform3D.setTranslation(gunMove);
        viewTransform3D.mul(gunTransform3D);
        gunTransformGroup.setTransform(viewTransform3D);
    }

    protected void doRotateZ(double radians, double theMove) {
        viewTransformGroup.getTransform(viewTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotZ(radians);
        viewTransform3D.mul(toMove);
        viewTransformGroup.setTransform(viewTransform3D);

        if (gunTransformGroup == null) return;
        Vector3d gunMove = new Vector3d(1, 1, 1);
        Transform3D gunTransform3D = new Transform3D();
        gunTransform3D.rotZ(radians * theMove);
        gunTransform3D.setTranslation(gunMove);
        viewTransform3D.mul(gunTransform3D);
        gunTransformGroup.setTransform(viewTransform3D);
    }

    protected void doMove(Vector3d theMove) {
        viewTransformGroup.getTransform(viewTransform3D);
        Transform3D theMoveTransform3d = new Transform3D();
        theMoveTransform3d.setTranslation(theMove);
        viewTransform3D.mul(theMoveTransform3d);
        // this is where you need to set the gun
        // transform group too:
        // gunTransformGroup.setTransform(transform3D);
        viewTransformGroup.setTransform(viewTransform3D);
        if (gunTransformGroup == null) return;


        Vector3d gunMove = new Vector3d(0, 0, 1);

        gunMove.add(theMove);

        Transform3D gunTransform3D = new Transform3D();
        gunTransform3D.setTranslation(gunMove);
        viewTransform3D.mul(gunTransform3D);
        gunTransformGroup.setTransform(viewTransform3D);
    }

    private void fireBullet() {
        // fire a bean from pp 9 of chapter 15 of fps
        // Get the transform group for the bullet.
        Transform3D tempT3d = new Transform3D();
        gunTransformGroup.getTransform(tempT3d);
        bulletTransformGroup.setTransform(tempT3d);
        //targetSwitch.setWhichChild(0);
        new RunJob(.75,false,100){
            public void run() {
                bulletMove();
               // System.out.println(getBulletTargetDistance());
                getBulletTargetDistance();
            }
        };
    }

    public void bulletMove() {
        Transform3D tempT3d = new Transform3D();
        bulletTransformGroup.getTransform(tempT3d);
        Transform3D toMove = new Transform3D();
        toMove.setTranslation(new Vector3d(0.0, 0, -0.50));
        tempT3d.mul(toMove);
        bulletTransformGroup.setTransform(tempT3d);
        // to debug, print the distance to the target each
        // time you move the bullet.
        // check to see if you are closeToTarget
        // if you are, then kapowski!
        // if not, just return.
        // if(closeToTarget()) blammo();
        // blammo targetSwitch.setWhichChild(1);
    }

    private boolean closeToTarget() {
        double sqLen = getBulletTargetDistance();
        if (sqLen < 1) return true;
        return false;
    }

   //  private boolean closeToTarget()
  /* The beam is close if its current position (currVec)
     is a short distance from the target position (targetVec).
  */
 // {
  //  beam.getLocalToVworld( localT3d );    // get beam's TG in world coords
  //  localT3d.get(currVec);                // get (x,y,z) component
    // System.out.println("currVec: " + currVec);

  //  currVec.sub(targetVec);    // distance between two positions
   // double sqLen = currVec.lengthSquared();
   // if (sqLen < HIT_RANGE*HIT_RANGE)
   //   return true;
  //  return false;
 // }  // end of closeToTarget()
   //SceneGraphObject.getLocalToVworld(Transform3D t)
   //       Retrieves the local coordinates to
   // // virtual world coordinates transform
   // for this node in the scene graph.
    private double getBulletTargetDistance() {
        Vector3d targetVector = new Vector3d();   // location of target
        Vector3d bulletVector = new Vector3d();
        Transform3D bulletTransform = new Transform3D();
       // bulletTransformGroup.getTransform(bulletTransform);
      bulletTransformGroup.getLocalToVworld( bulletTransform );
      System.out.println(bulletTransform);
      // get beam's TG in world coords
  //  localT3d.get(currVec);                // get (x,y,z) component
    // System.out.println("currVec: " + currVec);
        //System.out.println(bulletTransform);
        //          bulletTransform.get(bulletVector);
        // The following throws an exception, no right to read...
        //bulletTransformGroup.getLocalToVworld(bulletTransform);
        //bulletTransform.get(bulletVector)
        //------------------------------
        Transform3D targetTransform = new Transform3D();
        targetTransformGroup.getTransform(targetTransform);
        //targetTransformGroup.getLocalToVworld(targetTransform);

        targetTransform.get(targetVector);

        bulletVector.sub(targetVector);
        double sqLen = Math.sqrt(bulletVector.lengthSquared());
        //System.out.println(sqLen);
        return sqLen;
    }


    protected double getMovementRate() {
        return moveRate * speed;
    }

    protected double getRollLeftAmount() {
        return rotateZAmount * speed;
    }

    protected double getRollRightAmount() {
        return -rotateZAmount * speed;
    }

    protected double getRotateUpAmount() {
        return rotateYAmount * speed;
    }

    protected double getRotateDownAmount() {
        return -rotateYAmount * speed;
    }

    protected double getRotateLeftAmount() {
        return rotateYAmount * speed;
    }

    protected double getRotateRightAmount() {
        return -rotateYAmount * speed;
    }

    public void setRotateXAmount(double radians) {
        rotateXAmount = radians;
    }

    public void setRotateYAmount(double radians) {
        rotateYAmount = radians;
    }

    public void setRotateZAmount(double radians) {
        rotateZAmount = radians;
    }

    public void setMovementRate(double meters) {
        moveRate = meters; // Travel rate in meters/frame
    }

    public void setForwardKey(int key) {
        forwardKey = key;
    }

    public void setBackKey(int key) {
        backKey = key;
    }

    public void setLeftKey(int key) {
        leftKey = key;
    }

    public void setRightKey(int key) {
        rightKey = key;
    }
}
