package j3d.cr325;

import javax.media.j3d.*;
import javax.vecmath.Matrix3d;
import javax.vecmath.Vector3d;
import java.awt.*;
import java.awt.event.KeyEvent;
import java.util.Enumeration;

class GunBehavior extends Behavior {
    /**
     * The separate criteria that trigger this behaviour
     */
    protected WakeupCriterion theCriterion;
    /**
     * The alpha that is used to 'fire' the ball
     */
    protected Alpha theGunAlpha;
    /**
     * Used to animate the ball
     */
    protected PositionInterpolator theInterpolator;
    /**
     * Used to calculate the current direction of the gun
     */
    protected int aim = 0;
    /**
     * This is used to rotate the gun
     */
    protected TransformGroup aimXfmGrp;
    protected TransformGroup viewTransformGroup;
    /**
     * Used to aim the ball
     */
    protected Matrix3d aimShotMat = new Matrix3d();
    /**
     * Used to aim the gun
     */

    protected Matrix3d aimGunMat = new Matrix3d();
    protected Matrix3d viewMat = new Matrix3d();
    /**
     * Used to define the ball's direction
     */
    protected Transform3D aimShotXfm = new Transform3D();
    /**
     * Used to define the gun's direction
     */
    protected Transform3D aimGunXfm = new Transform3D();
    protected Transform3D viewTransform = new Transform3D();

    /**
     * Set up the data for the behaviour.
     *
     * @param a1        Alpha that drives the ball's animation.
     * @param pi        PositionInterpolator used for the ball.
     * @param gunRotGrp TransformGroup that is used to rotate the gun.
     * @param theBounds Bounds that define the active region for this behaviour.
     */
    public GunBehavior(BulletBehavior bb,
                       TransformGroup gunRotGrp, Bounds theBounds) {
        theGunAlpha = bb.getAlpha();
        theInterpolator = bb.getBulletInterpolator();
        setSchedulingBounds(theBounds);
        aimXfmGrp = gunRotGrp;
    }

    /**
     * This sets up the criteria for triggering the behaviour.
     * We simple want to wait for a key to be pressed.
     */
    public void initialize() {
        theCriterion = new WakeupOnAWTEvent(KeyEvent.KEY_PRESSED);
        wakeupOn(theCriterion);
    }

    /**
     * left key cursor rotate left, right cursor key
     * rotate right, spacebar fire.
     *
     * @param criteria
     */
    public void processStimulus(Enumeration criteria) {
        while (criteria.hasMoreElements()) {
            processInput(criteria);
        }
        wakeupOn(theCriterion);
    }

    private void processInput(Enumeration criteria) {
        WakeupCriterion theCriterion = (WakeupCriterion) criteria.nextElement();
        if (theCriterion instanceof WakeupOnAWTEvent) {
            AWTEvent[] triggers = ((WakeupOnAWTEvent) theCriterion).getAWTEvent();
            //Check if it's a keyboard event
            if (triggers[0] instanceof KeyEvent) {
                processKeyEvent(triggers);
            }
        }
    }

    private void processKeyEvent(AWTEvent[] triggers) {
        int keyPressed = ((KeyEvent) triggers[0]).getKeyCode();
        if (keyPressed == KeyEvent.VK_LEFT) {
            left();
        } else if (keyPressed == KeyEvent.VK_RIGHT) {
            right();

        } else if (keyPressed == KeyEvent.VK_UP) {
            //up();

        } else if (keyPressed == KeyEvent.VK_DOWN) {
            //down();
        } else if (keyPressed == KeyEvent.VK_SPACE) {
            //It's the spacebar so reset the start time
            //of the ball's animation
            theGunAlpha.setStartTime(System.currentTimeMillis());
        }
    }

    int x = 0;
    private void up() {
        translateGun(x);
        x++;
    }

    private void down(){
        translateGun(x);
        x--;
    }


    private void left() {
        //It's a left key so move the turret
        //and the aim of the gun left unless
        //we're at our maximum angle
        if (aim < 8)
            aim += 1;
        rotateGun();
    }

    private void translateGun(int x){
        aimShotMat.add(x);
        aimGunMat.add(x);
        final Vector3d v3d = new Vector3d(x,0,0);
        aimShotXfm.setTranslation(v3d);
        aimGunXfm.setTranslation(v3d);
        aimXfmGrp.setTransform(aimGunXfm);
        theInterpolator.setTransformAxis(aimShotXfm);
    }

    private void rotateGun() {
        aimShotMat.rotY(((aim / 32.0) + 0.5) * Math.PI);
        aimGunMat.rotZ(((aim / -32.0)) * Math.PI);
        aimShotXfm.setRotation(aimShotMat);
        aimGunXfm.setRotation(aimGunMat);
        aimXfmGrp.setTransform(aimGunXfm);
        theInterpolator.setTransformAxis(aimShotXfm);
    }

    private void right() {
        //It's the right key so do the same but rotate right
        if (aim > -8)
            aim -= 1;
        rotateGun();
    }
}


