package pavlik;

import futils.WriterUtil;
import gui.ClosableJFrame;
import gui.run.*;
import ip.ppm.WritePPM;
import j2d.ImageUtils;
import j2d.ShortImageBean;

import javax.swing.*;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.Graphics2D;
import java.awt.Graphics;
import java.awt.Dimension;
import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Image;
import java.awt.Point;
import java.awt.GridLayout;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.util.Vector;

/**
 * Created by IntelliJ IDEA.
 * User: David Pavlik
 * Date: May 8, 2004
 * Time: 1:48:30 PM
 */
public class AffineImagePanel extends JPanel implements MouseListener {
    private ClosableJFrame imageFrame;
    private Image image;
    private boolean rotateAboutMouse = false;

    private ClosableJFrame controlFrame;
    private Vector sliderLabels = getInitialSliderLabels();
    private Vector sliders = getInitialSliders();
    private JPanel labelPanel = getLabelPanel();
    private JPanel sliderPanel = getSliderPanel();
    private RunSlider translateX, translateY, rotate, scaleX, scaleY, shearX, shearY;

    private Point2D location = new Point2D.Float(0, 0);
    private Point2D scale = new Point2D.Double(1.0, 1.0);
    private Point2D shear = new Point2D.Double(0, 0);
    private double theta = 0;
    private Point2D mousePoint = new Point2D.Double(0, 0);

    private AffineTransform op;
    private AffineTransform lastTransform = new AffineTransform();;
    private AffineTransform priorTransform = new AffineTransform();

    public static void main(String[] args) {
        new AffineImagePanel();
    }

    public AffineImagePanel() {
        getNewImage();
        displayImageFrame();
        addMouseListener(this);
        displayControlFrame();
    }

    private void getNewImage() {
        image = ImageUtils.getImage();
        //Set the mousePoint to the image center
        mousePoint = new Point(image.getWidth(this) / 2, image.getHeight(this) / 2);
    }

    private void displayControlFrame() {
        controlFrame = new ClosableJFrame("Operations in Descending Order:");
        Container c = controlFrame.getContentPane();
        c.setLayout(new BorderLayout());
        controlFrame.setSize(350, 400);
        c.add(labelPanel, BorderLayout.WEST);
        c.add(sliderPanel, BorderLayout.CENTER);
        c.add(getButtonPanel(), BorderLayout.EAST);
        controlFrame.show();
    }

    private void displayImageFrame() {
        imageFrame = new ClosableJFrame();
        Container c = imageFrame.getContentPane();
        c.setLayout(new BorderLayout());
        c.add(getMenuBar(), BorderLayout.NORTH);
        c.add(this, BorderLayout.CENTER);
        int w = image.getWidth(this);
        int h = image.getHeight(this);
        imageFrame.setSize(w * 2, h * 2);
        imageFrame.show();
    }

    private JMenuBar getMenuBar() {
        JMenuBar mb = new JMenuBar();
        mb.add(getFileMenu());
        mb.add(getAffineMenu());

        return mb;
    }

    private RunMenu getFileMenu() {
        RunMenu rm = new RunMenu("[File");
        rm.add(new RunMenuItem("[Open... {ctrl O}") {
            public void run() {
                imageFrame.dispose();
                initializeSettings();
                getNewImage();
                displayImageFrame();
            }
        });
        JMenu saveAs = new JMenu("Save as... ");
        saveAs.setMnemonic(KeyEvent.VK_S);
        rm.add(saveAs);

        saveAs.add(new RunMenuItem("[GIF...(not 24 bit!)") {
            public void run() {
                saveAsGif();
            }
        });
        saveAs.add(new RunMenuItem("[PPM...") {
            public void run() {
                saveAsPPM();
            }
        });
        rm.add(new RunMenuItem("[Exit {ctrl E}") {
            public void run() {
                System.exit(0);
            }
        });

        return rm;
    }

    private void initializeSettings() {
        setSlidersToZero();
        priorTransform = new AffineTransform();
    }

    private void saveAsGif() {
        String fn = WriterUtil.getSaveFileName("Save as GIF");
        if (fn == null) return;
        saveAsGif(fn);
    }

    private void saveAsGif(String fn) {
        try {
            ip.vs.WriteGIF.DoIt(image, fn + ".gif");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void saveAsPPM() {
        String fn = WriterUtil.getSaveFileName("Save as PPM");
        if (fn == null) return;
        saveAsPPM(fn);
    }

    private void saveAsPPM(String fn) {
        ShortImageBean shortImageBean = new ShortImageBean();
        shortImageBean.setImage(image);
        WritePPM.doIt(shortImageBean.getR(), shortImageBean.getG(), shortImageBean.getB(), fn + ".ppm");
    }

    private RunMenu getAffineMenu() {
        RunMenu rm = new RunMenu("[Affine Menu");
        rm.add(new RunCheckBoxMenuItem("[Rotate About Mouse {ctrl R}") {
            public void run() {
                rotateAboutMouse = super.getState();
                if (!rotateAboutMouse) {
                    //Finalize the location of the lastTransform and make available as priorTransform
                    final AffineTransform fixedLastTransform = lastTransform;
                    priorTransform = fixedLastTransform;
                    //Set the mousePoint to the image center going through the priorTransform
                    mousePoint = priorTransform.transform(
                            new Point(image.getWidth(this) / 2, image.getHeight(this) / 2), null);
                    setSlidersToZero();
                }
            }
        });

        return rm;
    }

    private JPanel getLabelPanel() {
        JPanel jp = new JPanel();
        jp.setLayout(new GridLayout(0, 1));
        //Labels are added to the panel in the order they occur in the vector
        for (int i = 0; i < 7; i++) {
            jp.add((JLabel) sliderLabels.elementAt(i));
        }

        return jp;
    }

    private Vector getInitialSliderLabels() {
        Vector v = new Vector();
        v.add(new JLabel("Translate X"));
        v.add(new JLabel("Translate Y"));
        v.add(new JLabel("Rotate"));
        v.add(new JLabel("Scale X"));
        v.add(new JLabel("Scale Y"));
        v.add(new JLabel("Shear X"));
        v.add(new JLabel("Shear Y"));

        return v;
    }

    private JPanel getSliderPanel() {
        JPanel jp = new JPanel();
        jp.setLayout(new GridLayout(0, 1));
        //Sliders are added to the panel in the order they occur in the vector
        for (int i = 0; i < 7; i++) {
            jp.add((RunSlider) sliders.elementAt(i));
        }
        return jp;
    }

    private Vector getInitialSliders() {
        Vector v = new Vector();
        translateX = new RunSlider(-300, 300, 0) {
            public void run() {
                translateX(getValue());
            }
        };
        translateY = new RunSlider(-300, 300, 0) {
            public void run() {
                translateY(getValue());
            }
        };
        rotate = new RunSlider(-180, 180, 0) {
            public void run() {
                rotate(getValue());
            }
        };
        scaleX = new RunSlider(0, 200, 100) {
            public void run() {
                scaleX(getValue() / 100.0);
            }
        };
        scaleY = new RunSlider(0, 200, 100) {
            public void run() {
                scaleY(getValue() / 100.0);
            }
        };
        shearX = new RunSlider(-400, 400, 0) {
            public void run() {
                shearX(getValue() / 100.0);
            }
        };
        shearY = new RunSlider(-400, 400, 0) {
            public void run() {
                shearY(getValue() / 100.0);
            }
        };

        v.add(translateX);
        v.add(translateY);
        v.add(rotate);
        v.add(scaleX);
        v.add(scaleY);
        v.add(shearX);
        v.add(shearY);

        return v;
    }

    private JPanel getButtonPanel() {
        JPanel jp = new JPanel();
        jp.setLayout(new GridLayout(0, 1));
        jp.add(new RunButton("Set All To Zero") {
            public void run() {
                setSlidersToZero();
            }
        });
        //RunButtons are added to the panel that have an assigned sequential index
        //that causes the corresponding items in the sliderLabels and sliders vectors
        //to move ahead by one position in the vector. This becomes the new order
        //in which they appear in their controlFrame panels.
        for (int i = 1; i < 7; i++) {
            final int index = i;
            jp.add(new RunButton("Move Up") {
                int buttonIndex = index;

                public void run() {
                    sliderLabels.insertElementAt(sliderLabels.elementAt(buttonIndex), buttonIndex - 1);
                    sliderLabels.removeElementAt(buttonIndex + 1);
                    sliders.insertElementAt(sliders.elementAt(buttonIndex), buttonIndex - 1);
                    sliders.removeElementAt(buttonIndex + 1);
                    Container c = controlFrame.getContentPane();
                    c.removeAll();
                    c.add(getLabelPanel(), BorderLayout.WEST);
                    c.add(getSliderPanel(), BorderLayout.CENTER);
                    c.add(getButtonPanel(), BorderLayout.EAST);
                    controlFrame.show();
                }
            });
        }


        return jp;
    }

    private void setSlidersToZero() {
        translateX.setValue(0);
        translateY.setValue(0);
        rotate.setValue(0);
        scaleX.setValue(100);
        scaleY.setValue(100);
        shearX.setValue(0);
        shearY.setValue(0);
    }

    private void translateX(int x) {
        location.setLocation(x, location.getY());
        repaint();
    }

    private void translateY(int y) {
        location.setLocation(location.getX(), y);
        repaint();
    }

    private void scaleX(double sx) {
        scale.setLocation(sx, scale.getY());
        repaint();
    }

    private void scaleY(double sy) {
        scale.setLocation(scale.getX(), sy);
        repaint();
    }

    private void shearX(double sx) {
        shear.setLocation(sx, shear.getY());
        repaint();
    }

    private void shearY(double sy) {
        shear.setLocation(shear.getX(), sy);
        repaint();
    }

    private void rotate(double radians) {
        theta = radians;
        repaint();
    }

    public void mouseClicked(MouseEvent e) {
        if (!rotateAboutMouse) return;
        //Finalize the location of the lastTransform and make available as priorTransform
        final AffineTransform fixedPriorTransform = lastTransform;
        priorTransform = fixedPriorTransform;

        mousePoint = e.getPoint();
        setSlidersToZero();
        repaint();
    }

    public void mousePressed(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void paint(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        Dimension d = getSize();
        g2.setColor(Color.blue);
        g2.fillRect(0, 0, d.width, d.height);
        setUpAffineTransform(g2);
        if (image == null) return;
        g2.drawImage(image, 0, 0, this);
    }

    private void setUpAffineTransform(Graphics2D g2) {
        AffineTransform tr1 = new AffineTransform();
        AffineTransform tr2 = new AffineTransform();
        AffineTransform tr3 = new AffineTransform();

        tr1.translate(-mousePoint.getX(), -mousePoint.getY());
        tr1.concatenate(priorTransform);
        op = getOperation();
        op.concatenate(tr1);
        tr2.translate(mousePoint.getX(), mousePoint.getY());
        tr2.concatenate(op);
        tr3.translate(location.getX(), location.getY());
        tr3.concatenate(tr2);

        g2.setTransform(tr3);
        lastTransform = tr3;
    }

    private AffineTransform getOperation() {
        AffineTransform at = new AffineTransform();

        for (int i = 0; i < 7; i++) {
            if (((JLabel) sliderLabels.elementAt(i)).getText() == "Rotate")
                at.rotate(theta * Math.PI / 180);
            if (((JLabel) sliderLabels.elementAt(i)).getText() == "Scale X")
                at.scale(scale.getX(), 1);
            if (((JLabel) sliderLabels.elementAt(i)).getText() == "Scale Y")
                at.scale(1, scale.getY());
            if (((JLabel) sliderLabels.elementAt(i)).getText() == "Shear X")
                at.shear(shear.getX(), 0);
            if (((JLabel) sliderLabels.elementAt(i)).getText() == "Shear Y") at.shear(0, shear.getY());
        }

        return at;
    }
}

