package gui.run;

public abstract class RunJob
        implements Runnable {
    /**
     * The minimum priority that a RunJob can
     * have.
     */
    public static final int MIN_PRIORITY = 1;
    /**
     * The default priority that is assigned to a
     * RunJob.
     */
    public static final int NORM_PRIORITY = 5;
    /**
     * The maximum priority that a RunJob can
     * have.
     */
    public static final int MAX_PRIORITY = 10;
    private CommandThread t =
            new CommandThread(this);
    private long ms = 0;

    private boolean cowsComeHome = true;
    private int count = Integer.MIN_VALUE;
    private int numberOfTimesRun = 0;
    private String name = "runJob";
    private boolean isDaemon = false;
    private int priority = NORM_PRIORITY;


    /**
     * Runs the <code>doCommand()</command> every
     * <code>seconds</seconds>. The job starts
     * right away.
     * 
     * @param seconds 
     */
    public RunJob(double seconds) {
        this(seconds, false);
    }

    /**
     * Run the RunJob every <code> seconds</code>
     * If <code>wait</code> is <code>true</code>
     * then do not start the RunJob until the
     * start method is invoked. Otherwise, start
     * right away.
     * 
     * @param seconds 
     * @param wait    
     */
    public RunJob(double seconds, boolean wait) {
        this(seconds, wait, Integer.MIN_VALUE);
    }

    /**
     * Run the RunJob every <code> seconds</code>
     * If <code>wait</code> is <code>true</code>
     * then do not start the RunJob until the
     * start method is invoked. Otherwise, start
     * right away. If the <code>count</count> is
     * present, then the job only executes
     * <code>count</count>.
     * 
     * @param seconds 
     * @param wait    
     * @param count   
     */
    public RunJob(double seconds,
                  boolean wait,
                  int count) {
        this(seconds,wait,count,false);
    }

    /**
     * Run the RunJob every <code> seconds</code>
     * If <code>wait</code> is <code>true</code>
     * then do not start the RunJob until the
     * start method is invoked. Otherwise, start
     * right away. If the <code>count</count> is
     * present, then the job only executes
     * <code>count</count>. <code>isDaemon</code>
     * Marks this RunJob as
     * either a daemon RunJob or a user RunJob.
     * The Java Virtual Machine exits when the
     * only RunJobs running are all daemon
     * RunJobs and all Threads are daemon threads.
     * The default value for <code>isDaemon</code>
     * is <code>false</code>.
     *
     * @param seconds seconds of wait between jobs
     * @param wait   true if you want to delay start
     * @param count  number of times to execute
     * @param isDaemon true if you want to exit on main exit
     */
    public RunJob(double seconds,
                  boolean wait,
                  int count,
                  boolean isDaemon) {
        this.count = count;
        ms = (long) (seconds * 1000);
        if (wait == true) return;
        this.isDaemon = isDaemon;
        start();
    }

    /**
     * @return the number of times the RunJob was
     *         run.
     */
    public int getNumberOfTimesRun() {
        return numberOfTimesRun;
    }


    /**
     * Changes the priority of this RunJob.
     * <p/>
     * This may result in throwing a
     * <code>SecurityException</code>.
     * <p/>
     * Otherwise, the priority of this RunJob is
     * set to the smaller of the specified
     * <code>newPriority</code> and the maximum
     * permitted priority of the RunJobs's thread
     * group. If the priority is smaller than
     * <code>MIN_PRIORITY</code> then it will be
     * set to <code>MIN_PRIORITY</code>. If the
     * priority is greater than <code>MAX_PRIORITY</code>
     * then it will be set to <code>MAX_PRIORITY</code>.
     * This avoids illegal arguments.
     *
     * @param priority priority to set this RunJob
     *                 to
     * @throws SecurityException if the current
     *                           RunJob cannot
     *                           modify its thread.
     * @see RunJob#getPriority
     * @see #MAX_PRIORITY
     * @see #MIN_PRIORITY
     */
    public void setPriority(int priority) {
        if (priority > MAX_PRIORITY)
            priority = MAX_PRIORITY;
        if (priority < MIN_PRIORITY)
            priority = MIN_PRIORITY;
        this.priority = priority;
        t.setPriority(priority);
    }

    /**
     * Returns this RunJob's priority.
     * 
     * @return this RunJob's priority.
     * @see #setPriority
     * @see java.lang.Thread#setPriority(int)
     */
    public final int getPriority() {
        return t.getPriority();
    }

    /**
     * Start will restart a RunJob that was
     * stopped. It will also start a RunJob that
     * was never started.
     */
    public void start() {
        cowsComeHome = true;
        t = new CommandThread(this);
        t.setName(name);
        t.setDaemon(isDaemon());
        t.setPriority(priority);
        t.start();
    }

    /**
     * Changes the name of this RunJob (and thread
     * upon which the RunJob is based, to be equal
     * to the argument <code>name</code>.
     * <p/>
     * First the <code>checkAccess</code> method
     * of this thread is called with no arguments.
     * This may result in throwing a
     * <code>SecurityException</code>.
     * 
     * @param name the new name for this thread.
     */
    public void setName(String name) {
        this.name = name;
        t.setName(name);
    }

    /**
     * Gets the name of this RunJob (and the
     * thread upon which the RunJob is based
     * <p/>
     * First the <code>checkAccess</code> method
     * of this thread is called with no arguments.
     * This may result in throwing a
     * <code>SecurityException</code>.
     */
    public String getName() {
        return t.getName();
    }

    /**
     * Stops the RunJob by causing a thread death.
     * A <code>start()</code> invocation Will make
     * a new thread and restart the RunJob.
     */
    public void stop() {
        cowsComeHome = false;
    }

    /**
     * Tests if the RunJob is a daemon RunJob.
     * 
     * @return <code>true</code> if this RunJob is
     *         a daemon RunJob; <code>false</code>
     *         otherwise.
     */
    public boolean isDaemon() {
        return isDaemon;
    };
   

    public static void main(String args[]) {
        // anonymous inner class
        // That uses the command pattern
        // also uses facade pattern
        RunJob rj = new RunJob(2,false,5,false) {
            public void run() {
                System.out.println(
                        new java.util.Date());
            }
        };
        rj.start();
        rj.stop();
        rj.start();
    }

    /**
     * Tests if this RunJob is alive. A RunJob is
     * alive if it has been started and has not
     * yet run for the requisite number of times.
     * 
     * @return <code>true</code> if this RunJob is
     *         alive; <code>false</code>
     *         otherwise.
     */
    public boolean isAlive() {
        return t.isCountDone() && cowsComeHome;
    }

    /**
     * Returns a string representation of this
     * RunJob, including the RunJob's name,
     * priority, and thread group to which the
     * RunJob's thread belongs.
     * 
     * @return a string representation of this
     *         thread.
     */
    public String toString() {
        ThreadGroup group = t.getThreadGroup();
        if (group != null) {
            return "RunJob[" + getName() + "," +
                   getPriority() +
                   "," +
                   group.getName() + "]";
        } else {
            return "RunJob[" + getName() + "," +
                   getPriority() +
                   "," +
                   "" + "]";
        }
    }


    private class CommandThread extends Thread {
        RunJob job = null;


        CommandThread(RunJob job) {
            this.job = job;
        }


        private boolean isCountDone() {
            if (count == Integer.MIN_VALUE)
                return false;
            return numberOfTimesRun > count;
        }

        public void run() {
            while (cowsComeHome) {
                numberOfTimesRun++;
                if (isCountDone()) return;
                job.run();
                try {
                    Thread.sleep(ms);
                } catch (InterruptedException e) {
                }
            }
        }
    }

}
