package gui.dialogs;

import javax.swing.*;

/**
 * This class demonstrates the usage of the StandardDialog class.
 */
public class CustomerDialog extends ModalJDialog {
    // Constants

    private static final String[] COUNTRY_LIST =
            {
                "Australia", "England", "Japan", "USA"
            };

    // Attributes

    private JTextField myCustomerCodeField = new JTextField();
    private JTextField myNameField = new JTextField();
    private JTextArea myAddressField = new JTextArea(3, 20);
    private JTextField myCityField = new JTextField();
    private JTextField myStateField = new JTextField();
    private JTextField myPostcodeField = new JTextField();
    private JComboBox myCountryField = new JComboBox(COUNTRY_LIST);
    private JTextField myContactField = new JTextField();
    private JTextField myPhoneField = new JTextField();
    private JTextField myFaxField = new JTextField();
    private JTextField myEmailField = new JTextField();

    private LabelledItemPanel myContentPane = new LabelledItemPanel();

    // Methods

    /**
     * This method is the default constructor.
     */
    public CustomerDialog() {
        init();
    }

    /**
     * This method initialises the components on the panel.
     */
    private void init() {
        setTitle("Customer Dialog (StandardDialog)");

        myContentPane.setBorder(BorderFactory.createEtchedBorder());

        myContentPane.addItem("Customer Code", myCustomerCodeField);
        myContentPane.addItem("Name", myNameField);
        myContentPane.addItem("Address", new JScrollPane(myAddressField,
                JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                JScrollPane.HORIZONTAL_SCROLLBAR_NEVER));
        myContentPane.addItem("City", myCityField);
        myContentPane.addItem("State", myStateField);
        myContentPane.addItem("Postcode", myPostcodeField);
        myContentPane.addItem("Country", myCountryField);
        myContentPane.addItem("Contact", myContactField);
        myContentPane.addItem("Phone", myPhoneField);
        myContentPane.addItem("Fax", myFaxField);
        myContentPane.addItem("Email", myEmailField);

        setContentPane(myContentPane);
    }

    /**
     * This method gets the values of the panel entry fields.
     *
     * @return an object containing the Customer data
     */
    public CustomerData getCustomerData() {
        CustomerData customerData = new CustomerData();

        customerData.setCustomerCode(myCustomerCodeField.getText());
        customerData.setName(myNameField.getText());
        customerData.setAddress(myAddressField.getText());
        customerData.setCity(myCityField.getText());
        customerData.setState(myStateField.getText());
        customerData.setZip(myPostcodeField.getText());
        customerData.setCountry(myCountryField.getSelectedItem().toString());
        customerData.setContact(myContactField.getText());
        customerData.setPhone1(myPhoneField.getText());
        customerData.setFax(myFaxField.getText());
        customerData.setEmail(myEmailField.getText());

        return customerData;
    }

    /**
     * This method sets the values of the panel entry fields.
     *
     * @param customerData The object containing the Customer data
     */
    public void setCustomerData(CustomerData customerData) {
        myCustomerCodeField.setText(customerData.getCustomerCode());
        myNameField.setText(customerData.getName());
        myAddressField.setText(customerData.getAddress());
        myCityField.setText(customerData.getCity());
        myStateField.setText(customerData.getState());
        myPostcodeField.setText(customerData.getZip());
        myCountryField.setSelectedItem(customerData.getCountry());
        myContactField.setText(customerData.getContact());
        myPhoneField.setText(customerData.getPhone1());
        myFaxField.setText(customerData.getFax());
        myEmailField.setText(customerData.getEmail());
    }

    /**
     * This method checks that the data entered is valid.
     * To be valid, the following must be met:
     * <LI>Customer Code field is not blank
     * <LI>Name field is not blank
     *
     * @return <code>true</code> if the data is valid, otherwise
     * <code>false</code>
     */
    protected boolean isValidData() {
        if (myCustomerCodeField.getText().equals("")) {
            JOptionPane.showMessageDialog(this,
                    "Please enter a Customer Code",
                    "Blank Customer Code",
                    JOptionPane.WARNING_MESSAGE);

            myCustomerCodeField.requestFocus();

            return false;
        }

        if (myNameField.getText().equals("")) {
            JOptionPane.showMessageDialog(this,
                    "Please enter a Name",
                    "Blank Name",
                    JOptionPane.WARNING_MESSAGE);

            myNameField.requestFocus();

            return false;
        }

        return true;
    }

    /**
     * This method demonstrates the usage of the CustomerDialog class.
     */
    public static void main(String[] args) {
        CustomerDialog dialog = new CustomerDialog();

        dialog.pack();

        // Present it to a User

        dialog.show();

        // Get the data and display it

        if (!dialog.hasUserCancelled()) {
            CustomerData customerData = dialog.getCustomerData();

            customerData.printData();
        }

        System.exit(0);
    }
}