package classUtils.pack.util.jdbc;

import java.net.InetAddress;
import java.util.Properties;
import java.util.Set;

import classUtils.pack.util.SymbolTable;
import classUtils.pack.util.SymbolTable.UndefinedSymbolException;

/**
 * A base class for template JDBC URL templates.
 * <p>
 * This class must be provided with a template at construction (containing symbols are
 * defined in the <tt>$(<i>symbol</i>);</tt> form), then the corresponding values
 * must be filled in and a propre JDBC url can be obtained by {@link #getURL() getURL()}.
 * <p>
 * The following symbols:
 * <p>
 * <table border align=center>
 * <tr><td><b>subprotocol</b></td><td>the subprotocol, usually identifying a JDBC driver</td></tr>
 * <tr><td><b>host</b></td><td>the datbase host name</td></tr>
 * <tr><td><b>port</b></td><td>the datbase port</td></tr>
 * <tr><td><b>database</b></td><td>the datbase name</td></tr>
 * <tr><td><b>schema</b></td><td>the schema name</td></tr>
 * <tr><td><b>user</b></td><td>the username for connecting</td></tr>
 * <tr><td><b>password</b></td><td>the password for connecting</td></tr>
 * </table>
 * <p>
 * are directly supported. However, arbitrary symbols can be present in the template
 * and their values defined by {@link #setSymbolValue(java.lang.String, java.lang.String) 
 * setSymbolValue()}.
 * <p>
 * All the relative setters are <b>protected</b>. A subclass defines which symbol to 
 * expose publicly, simply by overriding one of the set<i>Symbol</i> methods with 
 * <b>public</b> visibility an whose body just invokes the corresponding method in this class.
 * @author Cristiano Sadun
 */
public class JdbcURL {
	
	private SymbolTable symbolTable;
	private String template;
	private Properties connectionProperties;
	private Set symbols;
		
	private JdbcURL(String template, Properties connectionProperties) {
		this.template=template;
		if (!template.startsWith("jdbc:")) throw new IllegalArgumentException("The given template is not a JDBC url");
		this.symbols=createSymbolsFromTemplate(template);
			this.connectionProperties=connectionProperties;
		this.symbolTable=new SymbolTable();
	}
	
	/**
	 * Create a jdbc URL object using a given template
	 */
	protected JdbcURL(String template) {
		this(template, new Properties());
	}
	
	/**
	 * Used to define a symbol in the template.
	 * <p>
	 * The symbol <i>must</i> be used in the template passed at construction.
	 */		
	protected void setSymbolValue(String symbol, String value) {
		if (! symbols.contains(symbol)) {
			throw new IllegalStateException("No "+symbol+" defined in the JDBC url template");
		}
		symbolTable.defineSymbol(symbol, value);
	}

	/**
	 * Define the standard "subprotocol" symbol.
	 */	
	protected void setSubprotocol(String subprotocol) {
		setSymbolValue("subprotocol", subprotocol);
	}


	/**
	 * Define the standard "database" symbol.
	 */	
	protected void setDatabase(String database) {
		setSymbolValue("database", database);
	}
	
	/**
	 * Define the standard "port" symbol.
	 */	
	protected void setPort(String port) {
		setSymbolValue("port", port);
	}
	
	/**
	 * Define the standard "port" symbol.
	 */		
	protected void setPort(int port) {
		setPort(String.valueOf(port));
	}
	
	/**
	 * Define the standard "host" symbol.
	 */		
	protected void setHost(String host) {
		setSymbolValue("host", host);
	}
	
	/**
	 * Define the standard "host" symbol.
	 */		
	protected void setHost(InetAddress host) {
		setSymbolValue("host", host.getHostAddress());
	}
	
	/**
	 * Define the standard "schema" symbol.
	 */		
	protected void setSchema(String schema) {
		setSymbolValue("schemaw", schema);
	}
	
	/**
	 * Define the standard "user" symbol.
	 */		
	protected void setUser(String username) {
		setSymbolValue("user", username);
	}	
	
	/**
	 * Define the standard "password" symbol.
	 */		
	protected void setPassword(String password) {
		setSymbolValue("password", password);
	}	
	
	/**
	 * Return the URL - substituting values to template symbols.
	 * @return the final JDBC URL
	 * @exception IllegalStateException if a symbol in the template is not defined
	 */
	public final String getURL() {
		try {
			return symbolTable.evaluate(template);
		} catch(SymbolTable.UndefinedSymbolException e) {
			throw new IllegalStateException("template symbol "+e.getSymbolName()+" has not been defined");
		}
	}

	/**
	 * Method createSymbolsFromTemplate.
	 */
	private static Set createSymbolsFromTemplate(String template) {
		SymbolTable st = new SymbolTable();
		st.setBehaviourOnUndefinedSymbol(st.RETURN_SYMBOL_ON_UNDEFINED_SYMBOL);
		st.evaluate(template);
		return st.getUndefinedSymbolsForLastEvaluation();
	}
}
