package sound;

import gui.ExtensionFileFilter;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.swing.*;
import java.awt.*;
import java.io.File;

public class AudioFileChooser {
    public static final short OPEN_DIALOG = 1;
    public static final short SAVE_DIALOG = 2;

    private File currentDirectory = null;
    private File selectedFile = null;
    private short type = OPEN_DIALOG;
    private AudioFormat sourceFormat = null;
    private AudioFormat.Encoding[] targetEncodingChoices = null;
    private AudioFormat.Encoding saveEncoding;

    /********************************************************/
    /**
     * Construct an AudioFileChooser which starts browsing the
     * file system at the specified directory.
     */
    public AudioFileChooser(File initialDirectory) {
        currentDirectory = initialDirectory;
    }

    /********************************************************/
    /**
     * Set chooser dialog type to OPEN_DIALOG or SAVE_DIALOG
     */
    public void setDialogType(short type)
            throws IllegalArgumentException {

        if (type != OPEN_DIALOG && type != SAVE_DIALOG) {
            throw new IllegalArgumentException("Incorrect dialog type specifier.");
        }

        this.type = type;
    }

    /********************************************************/
    /**
     * Display the dialog box and remember the user's choices.
     */
    public void show() {
        int chooserResult;

        JFileChooser chooser = new JFileChooser(currentDirectory);
        chooser.setMultiSelectionEnabled(false);
        ExtensionFileFilter filter1 = new ExtensionFileFilter(new String[]{"au", "wav"}, "All audio files");
        ExtensionFileFilter filter2 = new ExtensionFileFilter(new String[]{"au"}, "AU files");
        ExtensionFileFilter filter3 = new ExtensionFileFilter("wav", "WAV files");

        chooser.setAcceptAllFileFilterUsed(false);

        if (type == OPEN_DIALOG) {
            chooser.addChoosableFileFilter(filter1);
            chooser.addChoosableFileFilter(filter2);
            chooser.addChoosableFileFilter(filter3);
            chooser.setFileFilter(filter1);

            chooserResult = chooser.showOpenDialog(null);

            saveEncoding = null;
        } else {
            chooser.addChoosableFileFilter(filter2);
            chooser.addChoosableFileFilter(filter3);
            chooser.setFileFilter(filter2);

            JPanel extraPanel = new JPanel();
            JComboBox formatBox = new JComboBox();
            formatBox.addItem(new String("current encoding ("
                    + sourceFormat.getEncoding() + ")"));

            if (sourceFormat != null) {
                targetEncodingChoices = AudioSystem.getTargetEncodings(sourceFormat);

                for (int i = 0; i < targetEncodingChoices.length; i++) {
                    formatBox.addItem(targetEncodingChoices[i].toString());
                }
            }

            extraPanel.setLayout(new FlowLayout());
            extraPanel.add(new JLabel("Format:"));
            extraPanel.add(formatBox);
            chooser.add(extraPanel, BorderLayout.SOUTH);

            chooserResult = chooser.showSaveDialog(null);

            if (formatBox.getSelectedIndex() == 0) {
                saveEncoding = null;
            } else {
                saveEncoding = targetEncodingChoices[formatBox.getSelectedIndex() - 1];
            }
        }

        if (chooserResult == JFileChooser.APPROVE_OPTION) {

            currentDirectory = chooser.getCurrentDirectory();

            selectedFile = chooser.getSelectedFile();
        } else {
            selectedFile = null;
        }
    }

    /********************************************************/
    /**
     * Tell the AudioFileChooser about the AudioFormat of the
     * file being saved.
     */
    public void setSourceFormat(AudioFormat sourceFormat) {
        this.sourceFormat = sourceFormat;
    }

    /********************************************************/
    /**
     * Retrieve the directory path most recently used.
     */
    public File getCurrentDirectory() {
        return currentDirectory;
    }

    /********************************************************/
    /**
     * Get the file path of the selected file.
     */
    public File getSelectedFile() {
        return selectedFile;
    }

    /********************************************************/
    /**
     * Get the encoding selected in the save dialog.
     */
    public AudioFormat.Encoding getSelectedEncoding() {
        return saveEncoding;
    }
}


