package musica;
/**
 * Created by IntelliJ IDEA.
 * User: Carl
 * Date: Oct 3, 2003
 * Time: 3:34:29 PM
 * To change this template use Options | File Templates.
 */
public class MusicalScaleToMidi {
    public static final int midiMinNote = 0;
    public static final int midiMaxNote = 127;
    public static final int midiNoteRange = 128; // 0-127 possible midi notes
    public static final int midiOctaveSpan = 12; // intrinsic 12 tone chromatic intervalScale
    public static final int numMidiOctaves = 2 + midiNoteRange / midiOctaveSpan; //round up midi range on both ends

    private int[] intervalScale;// Programmer chooses type of intervalScale from Scales.java
    private int[] sequenceScale;// Scale sequence (ref to one octave) derived here from intervalScale
    private int lengthOfScale; // simplify clumsy references to array length
    private int userSequenceLo; // Lowest note in sequence scale, ref to 0 of scale
    private int userSequenceHi; // Highest note in sequence scale, ref to 0 of scale
    private int midiStartOfUserScale;  // Programmer chooses where to start interval Scale
    private int[][] midiUserScaleProgression;  // "keyboard" [octave][note] mapping to midi output
    // Program computes limits of octave-note inputs
    private int lowestUserOctave;  // Lowest within midi, in user octave numbers
    private int highestUserOctave; // Highest within midi, in user octave numbers
    private int lowestUserNote = midiMinNote;    // Lowest note within midi in lowest octave
    private int highestUserNote = midiMaxNote;  // Highest note within midi in highest octave
    /**
     * @param _intervalScale    choice of intervalScale from Scales class
     * @param _midiStartOfScale    insertion point of intervalScale[0] in midi 0-127
     */
    public MusicalScaleToMidi(int _intervalScale[], int _midiStartOfScale) {
        this.intervalScale = _intervalScale;   // Programmer responsible for reasonable numbers
        this.midiStartOfUserScale = _midiStartOfScale;    // 0 - 127
        lengthOfScale = intervalScale.length;
        sequenceScale = new int[intervalScale.length];
        computeSequenceScale();
        midiUserScaleProgression = new int[numMidiOctaves][lengthOfScale];
        computeUserToMidiOctaveOffsets();
        createNoteArray();
    }
    /**
     * Computes actual scale sequence from interval scale
     */
    private void computeSequenceScale() {
        sequenceScale[0] = intervalScale[0];
        for (int i = 1; i < intervalScale.length; i++) {
            sequenceScale[i] = sequenceScale[i - 1] + intervalScale[i];
        }
        userSequenceLo = sequenceScale[0];  // 1st note may not be at scale base (e.g. black notes)
        userSequenceHi = sequenceScale[lengthOfScale - 1]; // top note in scale, under octave top
    }

    private void computeUserToMidiOctaveOffsets() {
        int pointer = midiStartOfUserScale; //midi of user note 0 in any octave
        int counter = 0; // counts octave offset from user octave 0

        // First: Drop the pointer to below the midi range, if not there already
        while (pointer + userSequenceHi >= midiMinNote) {
            pointer -= midiOctaveSpan;
            counter--;
        }

        // Second: Raise pointer until first scale notes appear in midi range
        while (pointer + userSequenceHi < midiMinNote) {
            pointer += midiOctaveSpan;
            counter++;
        }
        lowestUserOctave = counter;
        int i = 0;
        while (pointer + sequenceScale[i] < midiMinNote) { // find first note above midiMin
            i++;
        }
        lowestUserNote = i;

        // Third: Raise the pointer above the midi range
        while (pointer + userSequenceLo <= midiMaxNote) {
            pointer += midiOctaveSpan;
            counter++;
        }

        // Fourth: Drop back within range
        pointer -= midiOctaveSpan;
        counter--;
        highestUserOctave = counter;
        i = lengthOfScale - 1; // point to top of scale
        while (pointer + sequenceScale[i] > midiMaxNote) { // find first note within range
            i--;
        }
        highestUserNote = i;
    }

    private void createNoteArray() {
        int tempNote;
        for (int arrayOctaveNumber = 0; arrayOctaveNumber < numMidiOctaves; arrayOctaveNumber++) {
            for (int noteNumber = 0; noteNumber < lengthOfScale; noteNumber++) {
                tempNote = mapToMidi(arrayOctaveNumber + lowestUserOctave, sequenceScale[noteNumber]);
                midiUserScaleProgression[arrayOctaveNumber][noteNumber] = tempNote;
            }
        }
    }

    private int mapToMidi(int playersOctaveNumber, int noteInOctave) {
        //  return midiNote;
        return midiStartOfUserScale + midiOctaveSpan * playersOctaveNumber + noteInOctave;
    }

    // Getters, toString, main
    public int[][] getMidiUserScaleProgression() {
        return midiUserScaleProgression;
    }

    public int[] getIntervalScale() {
        return intervalScale;
    }

    public int getMidiStartOfUserScale() {
        return midiStartOfUserScale;
    }

    public int getLowestUserOctave() {
        return lowestUserOctave;
    }

    public int getHighestUserOctave() {
        return highestUserOctave;
    }

    public int getLowestUserNote() {
        return lowestUserNote;
    }

    public int getHighestUserNote() {
        return highestUserNote;
    }

    public String toString() {
        String s = "";
        for (int octaveNumber = 0; octaveNumber < midiUserScaleProgression.length; octaveNumber++) {
            s += "\n";
            for (int noteNumber = 0; noteNumber < midiUserScaleProgression[0].length; noteNumber++) {
                s += midiUserScaleProgression[octaveNumber][noteNumber] + "\t";
            }
        }
        s += "\nmidiStartOfUserScale: " + midiStartOfUserScale;
        s+= "\nlowOctave; " + lowestUserOctave + "; \tlowNote = " + lowestUserNote;
        s+= "\nhighOctave; " + highestUserOctave + "; \thighNote = " + highestUserNote;
        return s;
    }

    public static void main(String[] args) {
        int startNote = 127;
        MusicalScaleToMidi pro =
                new MusicalScaleToMidi(ScaleManager.BLACK_KEYS, startNote);
        System.out.println(pro.toString());
    }
}