package j3d.cr325;

import com.sun.j3d.loaders.Scene;
import com.sun.j3d.loaders.objectfile.ObjectFile;
import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.geometry.Box;
import com.sun.j3d.utils.geometry.Cylinder;
import com.sun.j3d.utils.geometry.Sphere;
import j3d.Utils;

import javax.media.j3d.*;
import javax.vecmath.Color3f;
import javax.vecmath.Matrix3d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import java.applet.Applet;
import java.awt.*;
import java.io.FileNotFoundException;
import java.io.File;

/**
 * This application demonstrates a number of things in the
 * implementation of a simple shooting game. The object of the
 * the game is to shoot a duck that repeatedly moves across the
 * screen from left to right.  There are two duck models, one
 * for the 'live' duck and one for the 'dead' one.  These are
 * loaded from 'duck.obj' and 'deadduck.obj' files.  The 'gun'
 * is built from primitives.  The duck and the ball that is
 * used to shoot the duck use interpolators for their animation.
 * The gun uses key board input to aim and fire it, and
 * collision detection is used to 'kill' the duck.
 *
 * @author I.J.Palmer
 * @version 1.0
 */
public class SimpleGame extends Applet {

    private BoundingSphere bsBounds
            = new BoundingSphere(new Point3d(0.0, 0.0, 0.0), 100.0);
    /**
     * Alpha used to drive the duck animation
     */
    Alpha duckAlpha = new Alpha(-1, 0, 0, 3000, 0, 0);
    /**
     * Used to drive the ball animation
     */
    Alpha ballAlpha = new Alpha(1, 0, 0, 500, 0, 0);
    /**
     * Used to move the ball
     */
    PositionInterpolator ballPositionInterpolator = null;
    /**
     * Used to rotate the gun
     */
    TransformGroup gunXfmGrp = new TransformGroup();

    /**
     * This builds the view branch of the scene graph.
     *
     * @return BranchGroup with viewing objects attached.
     */
    protected static BranchGroup buildViewBranch(Canvas3D c) {
        BranchGroup viewBranch = new BranchGroup();
        Transform3D viewXfm = new Transform3D();
        Matrix3d viewTilt = new Matrix3d();
        viewTilt.rotX(Math.PI / -6);
        viewXfm.set(viewTilt, new Vector3d(0.0, 10.0, 10.0), 1.0);
        TransformGroup viewXfmGroup
                = new TransformGroup(viewXfm);
        ViewPlatform myViewPlatform
                = new ViewPlatform();
        PhysicalBody myBody
                = new PhysicalBody();
        PhysicalEnvironment myEnvironment
                = new PhysicalEnvironment();
        viewXfmGroup.addChild(myViewPlatform);
        viewBranch.addChild(viewXfmGroup);
        View myView = new View();
        myView.addCanvas3D(c);
        myView.attachViewPlatform(myViewPlatform);
        myView.setPhysicalBody(myBody);
        myView.setPhysicalEnvironment(myEnvironment);
        return viewBranch;
    }

    /**
     * This builds the gun geometry.
     * It uses box and cylinder primitives and sets up a
     * transform group so that we can rotate the gun.
     */
    protected BranchGroup buildGun() {
        BranchGroup theGun = new BranchGroup();
        Appearance gunApp = new Appearance();
        Color3f ambientColour = new Color3f(0.5f, 0.5f, 0.5f);
        Color3f emissiveColour = new Color3f(0.0f, 0.0f, 0.0f);
        Color3f specularColour = new Color3f(1.0f, 1.0f, 1.0f);
        Color3f diffuseColour = new Color3f(0.5f, 0.5f, 0.5f);
        float shininess = 20.0f;
        gunApp.setMaterial(new Material(ambientColour, emissiveColour,
                diffuseColour, specularColour, shininess));
        TransformGroup init = new TransformGroup();
        TransformGroup barrel = new TransformGroup();
        Transform3D gunXfm = new Transform3D();
        Transform3D barrelXfm = new Transform3D();
        barrelXfm.set(new Vector3d(0.0, -2.0, 0.0));
        barrel.setTransform(barrelXfm);
        Matrix3d gunXfmMat = new Matrix3d();
        gunXfmMat.rotX(Math.PI / 2);
        gunXfm.set(gunXfmMat, new Vector3d(0.0, 0.0, 0.0), 1.0);
        init.setTransform(gunXfm);
        gunXfmGrp.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        gunXfmGrp.addChild(new Box(1.0f, 1.0f, 0.5f, gunApp));
        barrel.addChild(new Cylinder(0.3f, 4.0f, gunApp));
        gunXfmGrp.addChild(barrel);
        theGun.addChild(init);
        init.addChild(gunXfmGrp);
        return theGun;
    }

    public static boolean exists(String s){
       File f = new File (s);
       return f.exists();
    }
    public static String getDataDir(String s) {
        if (exists(s)) return s;
        String f =  futils.Futil.getReadDirFile("could not find data directory:"
        +s
        +"Please locate another").toString();
        System.out.println("you selected:"+s);
        return getDataDir(s);
    }


    private Scene getDuckObject(String dataDir)
            throws FileNotFoundException {
        return new ObjectFile().load(dataDir + "duck.obj");
    }

    private Scene getDeadDuckObject(String dataDir)
            throws FileNotFoundException {
        return new ObjectFile().load(dataDir + "deadduck.obj");
    }


    /**
     * This builds the ball that acts as the bullet for our gun.
     * The ball is created from a sphere primitive, and a transform
     * group and interpolator are added so that we can 'fire' the
     * bullet.
     *
     * @return BranchGroup that is the root of the ball branch.
     */
    protected BranchGroup getBulletGroup() {
        BranchGroup bullet = new BranchGroup();

        Appearance bulletAppearance = new Appearance();
        Color3f ambientColour = new Color3f(1.0f, 0.0f, 0.0f);
        Color3f emissiveColour = new Color3f(0.0f, 0.0f, 0.0f);
        Color3f specularColour = new Color3f(1.0f, 1.0f, 1.0f);
        Color3f diffuseColour = new Color3f(1.0f, 0.0f, 0.0f);
        float shininess = 20.0f;
        bulletAppearance.setMaterial(
                new Material(ambientColour, emissiveColour,
                diffuseColour, specularColour, shininess));

        Sphere ball = new Sphere(0.2f, bulletAppearance);

        TransformGroup ballMovXfmGrp = new TransformGroup();
        ballMovXfmGrp.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        ballMovXfmGrp.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        ballMovXfmGrp.addChild(ball);
        bullet.addChild(ballMovXfmGrp);


        ballPositionInterpolator = new PositionInterpolator(ballAlpha,
                ballMovXfmGrp,
                new Transform3D(), 0.0f, 50.0f);
        ballPositionInterpolator.setSchedulingBounds(getBsBounds());
        bullet.addChild(ballPositionInterpolator);

        return bullet;

    }


    /**
     * This puts all the content togther.  It used the three 'build' functions
     * to create the duck, the gun and the ball.  It also creates the two
     * behaviours from the DuckBehaviour and GunBehaviour classes.  It then
     * puts all this together.
     *
     * @return BranchGroup that is the root of the content.
     */
    protected BranchGroup getContentBranch() {
        BranchGroup bg = new BranchGroup();
        Node theDuck = null;
        try {
            theDuck = Target.getTargetGroup();
        } catch (Exception e) {
            e.printStackTrace();
        }

        bg.addChild(theDuck);
        Node theBulletNode = getBulletGroup();
        bg.addChild(theBulletNode);
        DuckBehaviour hitTheDuck =
                new DuckBehaviour(theDuck, Target.getTargetSwitch(), duckAlpha, getBsBounds());
        GunBehaviour shootTheGun = new GunBehaviour(ballAlpha,
                ballPositionInterpolator,
                gunXfmGrp,
                getBsBounds());
        bg.addChild(hitTheDuck);
        bg.addChild(shootTheGun);
        bg.addChild(buildGun());
        Utils.addLights(getBsBounds(), bg);
        return bg;
    }


    private SimpleGame() {
        Canvas3D c3d
                = Utils.getCanvas3D(this);
        Locale l
                = new Locale(new VirtualUniverse());
        l.addBranchGraph(buildViewBranch(c3d));
        l.addBranchGraph(getContentBranch());
        setLayout(new BorderLayout());
        add("Center", c3d);
    }

    public static void main(String[] args) {
        new MainFrame(createSimpleGame(), 400, 400);
    }

    public static SimpleGame createSimpleGame() {
        return new SimpleGame();
    }

    public BoundingSphere getBsBounds() {
        return bsBounds;
    }

}
