// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.imageproc;

import java.awt.*;

/**
 * This classes allows adjustment of an image using histogram
 * equalization.
 * Reference: Douglas A. Lyon, "Image Processing in Java"
 */
public class HistogramEQProcessor extends ImageProcessor{

    private int lookupTable[] = new int[256];

    private double PMF[][] = new double[3][256];
    private double CMF[][] = new double[3][256];
    private double avgCMF[] = new double[256];

    private boolean blnExponential = false;
    private double dblAlpha = 0.001;

    /**
     * Create a new HistogramEQProcessor
     */
    public HistogramEQProcessor(){
        for (int j = 0; j< 256; j++){
            lookupTable[j] = j;
        }
    }

    /**
     * Implementation of ImageProcessor
     */
    public void performAlgorithm() throws Exception{
        int pixels[];
        int r, g, b;

        pixels = getPixels();

        for (int i = 0; i<pixels.length; i++){

            // Separate RGB components
            r = (pixels[i] & 0x00FF0000) >> 16;
            g = (pixels[i] & 0x0000FF00) >> 8;
            b = (pixels[i] & 0x000000FF);

            // Adjust the pixel
            r = lookupTable[r];
            g = lookupTable[g];
            b = lookupTable[b];

            // store the processed pixel
            pixels[i] = 0xFF000000 | (r << 16) | (g << 8) | b;
        }

        setPixels(pixels);
    }

    /**
     * Set the parameter for exponential EQ.
     *
     * @param alpha         parameter for exponential EQ
     */
    public void setAlpha(double alpha){
        if(alpha == dblAlpha) return;
        makeENAHETable(alpha);
        dblAlpha = alpha;
    }

    /**
     * Sets uniform/exponential EQ option.
     *
     * @param state         true = exponential; false = uniform
     */
    public void setExponential(boolean state){
        if (state == blnExponential) return;
        if (state == true){
            makeENAHETable(dblAlpha);
        }else{
            makeUNAHETable();
        }
        blnExponential = state;
    }

    /**
     * Set the base image reference.  Compute aggregate stats on
     * the image that will be needed for histogram processing.
     */
    public void setBaseImage(Image newImage){
        super.setBaseImage(newImage);
        computeStats();
        // compute lookup table
        if (blnExponential)
            makeENAHETable(dblAlpha);
        else
            makeUNAHETable();
    }

    /**
     * Compute PMF, CMF, etc.
     */
    private void computeStats(){
        int pixels[];
        int r, g, b;
        int i, j;
        double increment;

        pixels = getPixels();

        // Zero out all the stats.
        for (j = 0; j< 256; j++){
            for ( i= 0; i<3; i++){
                PMF[i][j] = 0;
                CMF[i][j] = 0;
            }
            avgCMF[j] = 0;
        }

        // Count lots of pixels

        increment = 1.0/pixels.length;
        for (i = 0; i<pixels.length; i++){

            // Separate RGB components
            r = (pixels[i] & 0x00FF0000) >> 16;
            g = (pixels[i] & 0x0000FF00) >> 8;
            b = (pixels[i] & 0x000000FF);

            PMF[0][r] += increment;
            PMF[1][g] += increment;
            PMF[2][b] += increment;
        }
        for (i = 0; i< 3; i++){
            CMF[i][0] = PMF[i][0];
            for (j = 1; j < 256; j++){
                CMF[i][j] = CMF[i][j-1] + PMF[i][j];
            }
        }
        for (i = 0; i < 3; i++){
            for (j=0; j < 256; j++){
                avgCMF[j] += CMF[i][j] / 3;
            }
        }
    }

    /**
     * Create lookup table for Uniform Non Adaptive
     * Histogram Equalization
     */
    private void makeUNAHETable() {
        for (short i = 0; i < lookupTable.length; i++){
            lookupTable[i] = (short) (255 * avgCMF[i]);
        }
    }

    /**
     * Create lookup table for Exponential Non Adaptive
     * Histogram Equalization
     */
    private void makeENAHETable(double alpha) {
        double val;
        double inverseCMF;

        for (short i = 0; i < lookupTable.length; i++){
            inverseCMF = 1.0 - avgCMF[i];
            if (inverseCMF <= 0){
                // Keep log in defined range at top of CMF; the CMF
                // may be >1 due to rounding.
                val = 255;
            }else{
                val = 255.0 * (-Math.log(inverseCMF) / alpha);
                val = Math.min(255, val);
                val = Math.max(0, val);
            }
            lookupTable[i] = (int)val;
        }
    }
}
