// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.gui;


import utils.ObservableFloat;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.text.DecimalFormat;
import java.util.Hashtable;
import java.util.Observer;

/**
 * A slider with floating point values, a name label, and built-in
 * tick labels.
 */
public class ObservableSlider extends JPanel {
    private final int JSLIDER_MIN = 0;
    private final int JSLIDER_MAX = 1000;

    private JSlider js;
    private JLabel lblDescription;
    private JLabel lblValue;
    private JPanel pnlLabels;

    private Hashtable hshLabelTable;
    private ObservableFloat sliderPosition;
    private DecimalFormat df;
    private float sliderMin = 0.0f;
    private float sliderMax = 100.0f;
    private boolean isLogScale = false;

    /**
     * Construct a new ObservableSlider ranging from 0.0f to 100.0f
     */
    public ObservableSlider() {
        hshLabelTable = new Hashtable();
        sliderPosition = new ObservableFloat();
        df = new DecimalFormat("##0.000");
        setLayout(new BorderLayout());

        js = new JSlider(SwingConstants.VERTICAL, JSLIDER_MIN,
                JSLIDER_MAX, JSLIDER_MIN);
        js.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                float val = intToDouble(js.getValue());
                sliderPosition.setValue(val);
                lblValue.setText(df.format(val));
            }
        });
        add(js, BorderLayout.CENTER);

        lblDescription = new JLabel("", SwingConstants.CENTER);
        lblValue = new JLabel(df.format(sliderMin));
        pnlLabels = new JPanel(new GridLayout(0, 1));
        pnlLabels.add(lblDescription);
        pnlLabels.add(lblValue);
        add(pnlLabels, BorderLayout.SOUTH);
        sliderPosition.setValue(intToDouble(JSLIDER_MIN));
    }

    /**
     * Construct a ObservableSlider ranging from min to max
     *
     * @param min The minimum value of the slider
     * @param max The maximum value of the slider
     */
    public ObservableSlider(float min, float max) {
        this();
        setMinimum(min);
        setMaximum(max);
    }

    /**
     * Set the text of the label.
     *
     * @param s The string which will appear below the slider.
     */
    public void setLabel(String s) {
        lblDescription.setText(s);
    }

    /**
     * Set the label of a tick
     *
     * @param value The tick at which the label will appear.
     * @param label The text of the label
     */
    public void setTickLabel(float value, String label) {
        hshLabelTable.put(new Integer(doubleToInt(value)), new JLabel(label));
        js.setLabelTable(hshLabelTable);
        js.setPaintLabels(true);
    }

    /**
     * Get the minimum value of the slider.
     *
     * @return Minimum slider value
     */
    public float getMinimum() {
        return sliderMin;
    }

    /**
     * Set the minimum value of the slider.
     *
     * @param newValue New minimum slider value
     */
    public void setMinimum(float newValue) {
        float position;
        sliderMin = newValue;
        position = Math.max(sliderPosition.getValue(), newValue);
        js.setValue(doubleToInt(position));
        sliderPosition.setValue(position);
        lblValue.setText(df.format(position));
    }

    /**
     * Get the maximum value of the slider.
     *
     * @return Maximum slider value
     */
    public float getMaximum() {
        return sliderMax;
    }

    /**
     * Set the maximum value of the slider.
     *
     * @param newValue New maximum slider value
     */
    public void setMaximum(float newValue) {
        float position;
        sliderMax = newValue;
        position = Math.min(sliderPosition.getValue(), newValue);
        js.setValue(doubleToInt(position));
        sliderPosition.setValue(position);
        lblValue.setText(df.format(position));
    }

    /**
     * Determine whether the slider has a log scale.
     *
     * @return True = log; false = linear
     */
    public boolean isLogScale() {
        return isLogScale;
    }

    /**
     * Set whether the slider has a log scale.
     *
     * @param state True = log; false = linear
     */
    public void setLogScale(boolean state) {
        isLogScale = state;
    }

    /**
     * Get the current value of the slider.
     *
     * @return The floating point value of the slider.
     */
    public float getValue() {
        return sliderPosition.getValue();
    }

    /**
     * Set the current value of the slider.
     *
     * @param newValue The new floating point position value.
     */
    public void setValue(float newValue) {
        js.setValue(doubleToInt(newValue));
    }

    /**
     * Change enabled/disabled state.
     *
     * @param state True = enabled, false = disabled
     */
    public void setEnabled(boolean state) {
        js.setEnabled(state);
        lblDescription.setEnabled(state);
        lblValue.setEnabled(state);
    }

    /**
     * Add the specified Observer to the list of parties notified
     * if the slider is moved.
     *
     * @param o The Observer to be added.
     */
    public void addObserver(Observer o) {
        sliderPosition.addObserver(o);
    }

    /**
     * Convert the integer index of the JSlider to a floating point
     * number in the user's specified range.
     *
     * @param in A position on the underlying JSlider.
     * @return The position in the user-specified range.
     */
    private float intToDouble(int in) {
        if (!isLogScale) {
            return sliderMin + (sliderMax - sliderMin) / (JSLIDER_MAX - JSLIDER_MIN) *
                    (in - JSLIDER_MIN);
        } else {
            return 0; //TODO - write this equation
        }
    }

    /**
     * Convert a floating point number in the user's specified range
     * to an integer index on the underlying JSlider.
     *
     * @param in A position in the user-specified range.
     * @return The position on the underlying JSlider.
     */
    private int doubleToInt(float in) {
        if (!isLogScale) {
            return JSLIDER_MIN + (int) (((float) JSLIDER_MAX - JSLIDER_MIN)
                    / (sliderMax - sliderMin) * (in - sliderMin));
        } else {
            return 0; //TODO - write this equation
        }
    }
}