// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.gui;

// Standard libraries

import j2d.file.ExtensionFileFilter;
import j2d.file.PPMToolkit;
import j2d.imageproc.ExponentialStretchProcessor;
import j2d.imageproc.FalseColorProcessor;
import j2d.imageproc.HistogramEQProcessor;
import j2d.imageproc.LinearMappingProcessor;

import javax.swing.*;
import javax.media.format.VideoFormat;
import javax.media.protocol.PushBufferDataSource;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import java.util.Observable;
import java.util.Observer;

import gui.run.RunMenuItem;
import gui.run.RunMenu;
import gui.ClosableJFrame;
import futils.StreamSniffer;
import j2d.ImageUtils;

/**
 * This class provides a GUI for altering images
 */

public class MainFrame extends ClosableJFrame {

    private MDIDesktopPane deskTop;

    private FalseColorToolbox falseColorToolbox;
    private LinearMappingToolbox linearMappingToolbox;
    private HistogramEQToolbox histEqToolbox;
    private ExponentialStretchToolbox estStretchControls;


    // Image processors

    private FalseColorProcessor fcProcessor
            = new FalseColorProcessor();
    private LinearMappingProcessor lmProcessor
            = new LinearMappingProcessor();
    private HistogramEQProcessor heProcessor
            = new HistogramEQProcessor();
    private ExponentialStretchProcessor esProcessor
            = new ExponentialStretchProcessor();

    // Capture details
    private VideoFormat vfFormat;
    private PushBufferDataSource pbds;

    /**
     * Construct a new ImageTool
     */
    public MainFrame() {
        Container c;

// Set up the frame surface.

        c = getContentPane();
        c.setLayout(new BorderLayout());

        deskTop = new MDIDesktopPane();
        c.add(new JScrollPane(deskTop));

        setTitle("Image Tool");
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        // Set up the menus

        JMenuBar mb = new JMenuBar();
        setJMenuBar(mb);
        addFileMenu(mb);
        addProcessMenu(mb);
        WindowMenu windowMenu = new WindowMenu(deskTop);
        mb.add(windowMenu);
        windowMenu.setMnemonic('W');
    }

    private void addProcessMenu(JMenuBar mb) {
        RunMenu processMenu = new RunMenu("[process");
        mb.add(processMenu);
        processMenu.add(new RunMenuItem("[False Color") {
            public void run() {
                if (falseColorToolbox == null || falseColorToolbox.isClosed()) {
                    launchColorControls();
                }
            }
        });
        processMenu.add(new RunMenuItem("[negate{ctrl N}") {
            public void run() {
                Utils.negate(deskTop);
            }
        });
        processMenu.add(new RunMenuItem("[Linear Mapping") {
            public void run() {
                if (linearMappingToolbox == null || linearMappingToolbox.isClosed()) {
                    launchMappingControls();
                }
            }
        });
        processMenu.add(new RunMenuItem("[Histogram EQ{ctrl H}") {
            public void run() {
                if (histEqToolbox == null || histEqToolbox.isClosed()) {
                    launchHistogramControls();
                }
            }
        });
        processMenu.add(new RunMenuItem("[Exponential Stretch") {
            public void run() {
                if (estStretchControls == null || estStretchControls.isClosed()) {
                    launchStretchControls();
                }
            }
        });
    }

    private void addFileMenu(JMenuBar mb) {
        RunMenu fileMenu = new RunMenu("[File");
        mb.add(fileMenu);

        fileMenu.add(
                new RunMenuItem("[Open", ImageUtils.fetchIcon(this, "icon_open.gif")) {
                    public void run() {
                        openImage();
                    }
                });


        fileMenu.add(new RunMenuItem("[Save{ctrl S}",
                ImageUtils.fetchIcon(this, "icon_save.gif")) {
            public void run() {
                saveImage();
            }
        });
        fileMenu.add(new RunMenuItem("[Revert{crtl R}",
                ImageUtils.fetchIcon(this, "icon_revert.gif")) {
            public void run() {
                revertImage();
            }
        });
        fileMenu.add(new RunMenuItem("[Exit{ctrl Q}",
                ImageUtils.fetchIcon(this, "icon_exit.gif")) {
            public void run() {
                shutdown();
            }
        });
    }

    /**
     * Helper function for closing the ImageTool.  Could put
     * other termination code here.
     */
    private void shutdown() {
        dispose();
    }

    /**
     * This is called when the user requests a new file to be opened.
     */
    private void openImage() {
        File imageFile;
        Image imgProcessee;
        ImageChildFrame icfNewFrame;
        int intSuccess = 0;

        imageFile = futils.Futil.getReadFile("select an image file");
        imgProcessee = getImage(imageFile);

        icfNewFrame = new ImageChildFrame(imgProcessee, imageFile.getName());
        deskTop.add(icfNewFrame);

    }

    public Image getImage(File f) {
        if (f == null) return null;
        if (!f.exists()) return null;

        InputStream is = null;
        try {
            is = new FileInputStream(f);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        StreamSniffer ss = new StreamSniffer(is);
        int id = ss.classifyStream();
        switch (id) {
            case StreamSniffer.PPM:
                return getPPMImage(f);
            case StreamSniffer.PPM_RAWBITS:
                return getPPMImage(f);
            case StreamSniffer.GIF87a:
                return getGifOrJpg(f);
            case StreamSniffer.GIF89a:
                return getGifOrJpg(f);
            case StreamSniffer.JPEG:
                return getGifOrJpg(f);
            default:
                {
                    System.out.println(
                            "Can not open " +
                            ss +
                            " as image");
                    return null;
                }

        }
    }

    private Image getGifOrJpg(File imageFile) {
        return Toolkit.getDefaultToolkit().getImage(
                imageFile.getAbsolutePath());
    }


    private Image getZippedPPMImage(File imageFile) {
        Image imgProcessee = null;
        try {
            imgProcessee = PPMToolkit.getImage(imageFile, true);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return imgProcessee;
    }

    private Image getPPMImage(File imageFile) {
        try {
            Image imgProcessee;
            imgProcessee = PPMToolkit.getImage(imageFile, false);
            return imgProcessee;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * This is called when the user picks the save menu option.
     */
    private void saveImage() {
        ImageChildFrame icf;
        int intSuccess;
        File imageFile;


        try {
            icf = (ImageChildFrame) deskTop.getTopmostFrame(ImageChildFrame.class);
            if (icf == null) return;
        } catch (Exception e) {
            return;
        }

        JFileChooser jfc = new JFileChooser();
        ExtensionFileFilter eff1 =
                new ExtensionFileFilter("ppm", "Portable Pixelmap image file");
        jfc.addChoosableFileFilter(eff1);
        ExtensionFileFilter eff2 =
                new ExtensionFileFilter("ppm.gz", "Zipped Pixelmap image file");
        jfc.addChoosableFileFilter(eff2);
        jfc.setFileFilter(eff1);
        jfc.setAcceptAllFileFilterUsed(false);

        intSuccess = jfc.showSaveDialog(this);

        if (intSuccess == JFileChooser.APPROVE_OPTION) {
            imageFile = jfc.getSelectedFile();

            ExtensionFileFilter eff = (ExtensionFileFilter) jfc.getFileFilter();

            try {
                if (eff == eff1) {
                    imageFile = fixExtension(imageFile, "ppm");
                    PPMToolkit.saveImage(icf.getImage(), imageFile, false);
                } else if (eff == eff2) {
                    imageFile = fixExtension(imageFile, "ppm.gz");
                    PPMToolkit.saveImage(icf.getImage(), imageFile, true);
                }
                icf.setTitle(imageFile.getName());
            } catch (Exception e) {
                JOptionPane.showMessageDialog(this, "File write error: \n"
                        + e.toString());
            }
        }
    }

    /**
     * Fix the file extension
     */
    private File fixExtension(File file, String extension) {
        String path = file.getAbsolutePath();

        if (!path.endsWith("." + extension)) {
            path += "." + extension;
        }
        return new File(path);
    }

    /**
     * Set the image in the active frame to the base image.
     */
    private void revertImage() {
        ImageChildFrame icf;

        try {
            icf = (ImageChildFrame)
                    deskTop.getTopmostFrame(
                            ImageChildFrame.class);
            icf.revert();
        } catch (Exception e) {
            // do nothing
        }
    }

    /**
     * Helper function to show LinearMappingToolbox
     */
    private void launchMappingControls() {
        linearMappingToolbox = new LinearMappingToolbox();
        linearMappingToolbox.show();
        linearMappingToolbox.getSliderBank().addObserver(new Observer() {
            public void update(Observable x, Object y) {
                adjustLinearMapping(false);
            }
        });
        linearMappingToolbox.getButton().addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                adjustLinearMapping(true);
            }
        });
        deskTop.add(linearMappingToolbox);
    }

    /**
     * Helper function to show FalseColorToolbox
     */
    private void launchColorControls() {
        falseColorToolbox = new FalseColorToolbox();
        falseColorToolbox.show();
        falseColorToolbox.getSliderBank().addObserver(new Observer() {
            public void update(Observable x, Object y) {
                adjustColorization();
            }
        });
        deskTop.add(falseColorToolbox);
    }

    /**
     * Helper function to show HistogramEQToolbox
     */
    private void launchHistogramControls() {
        histEqToolbox = new HistogramEQToolbox();
        histEqToolbox.show();
        histEqToolbox.getButton().addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                adjustHistogram();
            }
        });
        histEqToolbox.getSlider().addObserver(new Observer() {
            public void update(Observable x, Object y) {
                adjustHistogram();
            }
        });

        deskTop.add(histEqToolbox);
    }

    /**
     * Helper function to show ExponentialStretchToolbox
     */
    private void launchStretchControls() {
        estStretchControls = new ExponentialStretchToolbox();
        estStretchControls.show();
        estStretchControls.getSlider().addObserver(new Observer() {
            public void update(Observable x, Object y) {
                adjustExponentialStretch();
            }
        });
        deskTop.add(estStretchControls);
    }


    /**
     * This is called each time a slider in the FalseColorToolbox is moved.
     * This is where the real image processing happens.
     */
    private void adjustColorization() {
        float coeff[] = new float[3];
        ImageChildFrame icf;

        try {
            coeff = falseColorToolbox.getSliderBank().getValues();

            icf = (ImageChildFrame) deskTop.getTopmostFrame(ImageChildFrame.class);
            fcProcessor.setBaseImage(icf.getBaseImage());
            fcProcessor.setColorization(coeff[0], coeff[1], coeff[2]);
            fcProcessor.performAlgorithm();
            icf.setImage(fcProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called each time a slider on the LinearMappingToolbox is moved
     * (btn = false) or the button on the LinearMappingToolbox is pressed
     * (btn = true).  This is where the real image processing happens.
     */
    private void adjustLinearMapping(boolean btn) {
        float coeff[] = new float[2];
        ImageChildFrame icf;

        try {
            icf = (ImageChildFrame) deskTop.getTopmostFrame(ImageChildFrame.class);
            lmProcessor.setBaseImage(icf.getBaseImage());
            if (btn) {
// compute parameters from the image, then update the toolbox sliders
                coeff = lmProcessor.setOptimalParameters();
                linearMappingToolbox.getSliderBank().setValues(coeff);
            } else {
// get the parameters from the sliders, then update the image
                coeff = linearMappingToolbox.getSliderBank().getValues();
                lmProcessor.setParameters(coeff[0], coeff[1]);
            }
            lmProcessor.performAlgorithm();
            icf.setImage(lmProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called when the slider on the HistogramEQToolbox is
     * moved, or when its button is pressed.
     */
    private void adjustHistogram() {
        boolean exponential;
        float alpha;
        ImageChildFrame icf;

        try {
            icf = (ImageChildFrame) deskTop.getTopmostFrame(ImageChildFrame.class);
            heProcessor.setBaseImage(icf.getBaseImage());

            exponential = histEqToolbox.getExponential();
            heProcessor.setExponential(exponential);
            if (exponential) {
                alpha = histEqToolbox.getSlider().getValue();
                heProcessor.setAlpha(alpha);
            }

            heProcessor.performAlgorithm();
            icf.setImage(heProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called each time a slider in the ExponentialStretchToolbox is moved.
     * This is where the real image processing happens.
     */
    private void adjustExponentialStretch() {
        float coeff;
        ImageChildFrame icf;
        try {
            coeff = estStretchControls.getSlider().getValue();

            icf = (ImageChildFrame) deskTop.getTopmostFrame(ImageChildFrame.class);
            esProcessor.setBaseImage(icf.getBaseImage());
            esProcessor.setPower(coeff);
            esProcessor.performAlgorithm();
            icf.setImage(esProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * Entry point of the ImageTool application.
     */
    public static void main(String args[]) {
        MainFrame fc = new MainFrame();
        fc.setSize(500, 400);
        fc.show();
    }
}