package ip.transforms;

import math.Mat2;


public final class Kernels {
    private Kernels() {
    }

    private static double robinsonScaleFactor = .25;

    // Robinson level 5 detector, see Pratt.
    public static float[][] getRobinson1() {
        float k[][] = {
            {1, 0, -1},
            {2, 0, -2},
            {1, 0, -1}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getRobinson2() {
        float k[][] = {
            {0, -1, -2},
            {1, 0, -1},
            {2, 1, 0}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getSobel() {
        float k[][] = {
            {-1, -1, -1},
            {-1, 8, -1},
            {-1, -1, -1}
        };
        return k;
    }

    public static float[][] getSobelX(int a) {
        float k[][] = {
            {-1, 0, 1},
            {-a, 0, a},
            {-1, 0, 1}
        };
        return k;
    }

    public static float[][] getSobelY(int a) {
        float k[][] = {
            {-1, -a, 1},
            {0, 0, 0},
            {1, a, 1}
        };
        return k;
    }


    public static float[][] getRobinson3() {
        float k[][] = {
            {-1, -2, -1},
            {0, 0, 0},
            {1, 2, 1}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getRobinson4() {
        float k[][] = {
            {-2, -1, 0},
            {-1, 0, 1},
            {0, 1, 2}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getRobinson5() {
        float k[][] = {
            {-1, 0, 1},
            {-2, 0, 2},
            {-1, 0, 1}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getRobinson6() {
        float k[][] = {
            {0, 1, 2},
            {-1, 0, 1},
            {-2, -1, 0}
        };
        Mat2.scale(k, 0.25);
        return k;
    }

    public static float[][] getRobinson7() {
        float k[][] = {
            {1, 2, 1},
            {0, 0, 0},
            {-1, -2, -1}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }

    public static float[][] getRobinson8() {
        float k[][] = {
            {2, 1, 0},
            {1, 0, -1},
            {0, -1, -2}
        };
        Mat2.scale(k, robinsonScaleFactor);
        return k;
    }


    public static short[][] getMedian2x1() {
        short k[][] = {
            {0, 1, 0},
            {0, 1, 0},
            {0, 0, 0}
        };
        return k;
    }

    public static short[][] getMedian1x2() {
        short k[][] = {
            {0, 0, 0},
            {1, 1, 0},
            {0, 0, 0}
        };
        return k;
    }

    // p0 p1 p2
    // p3 p4 p5
    // p6 p7 p8
    //
    //
    public static short[][] getSizeDetector(short f[][]) {
        short a[][] = new short[f.length][f[0].length];
        int p[] = new int[9];
        int sum = 0;
        for (int x = 1; x < f.length - 1; x++)
            for (int y = 1; y < f[0].length - 1; y++) {
                sum = 0;
                p[0] = f[x - 1][y + 1];
                p[1] = f[x][y + 1];
                p[2] = f[x + 1][y + 1];
                p[3] = f[x - 1][y];
                p[4] = f[x][y];
                p[5] = f[x + 1][y];
                p[6] = f[x - 1][y - 1];
                p[7] = f[x][y - 1];
                p[8] = f[x + 1][y - 1];
                for (int i = 0; i < p.length; i++)
                    sum += p[i];
                if (sum > 255 * 5)
                    a[x][y] = 255;
                else
                    a[x][y] = 0;
            }
        return a;
    }

    public static float[][] getLaplacian5() {
        float k[][] = {
            {-1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1},
            {-1, -1, 24, -1, -1},
            {-1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1}
        };
        return k;
    }

    public static float[][] getLaplacian3() {
        float k[][] = {
            {0, -1, 0},
            {-1, 4, -1},
            {0, -1, 0}
        };
        return k;
    }

    public static float[][] getLaplacianPrewitt() {
        float k[][] = {
            {-1, -1, -1},
            {-1, 8, -1},
            {-1, -1, -1}
        };
        return k;
    }

    public static double laplaceOfGaussian(double x, double y,
                                           double xc, double yc, double sigma) {
        ConvolutionUtils.t1 = sigma * sigma;
        ConvolutionUtils.t2 = ConvolutionUtils.t1 * ConvolutionUtils.t1;
        ConvolutionUtils.t5 = Math.pow(x - xc, 2.0);
        ConvolutionUtils.t7 = Math.pow(y - yc, 2.0);
        ConvolutionUtils.t11 = Math.exp(-(ConvolutionUtils.t5 + ConvolutionUtils.t7) / ConvolutionUtils.t1 / 2);
        ConvolutionUtils.t13 = 1 / Math.PI;
        ConvolutionUtils.t16 = Math.pow(2.0 * x - 2.0 * xc, 2.0);
        ConvolutionUtils.t18 = 1 / ConvolutionUtils.t2 / ConvolutionUtils.t1;
        ConvolutionUtils.t20 = ConvolutionUtils.t11 * ConvolutionUtils.t13;
        ConvolutionUtils.t23 = Math.pow(2.0 * y - 2.0 * yc, 2.0);
        ConvolutionUtils.t26 = 1 / ConvolutionUtils.t2 * ConvolutionUtils.t11 * ConvolutionUtils.t13 - ConvolutionUtils.t16 * ConvolutionUtils.t18 * ConvolutionUtils.t20 / 8 - ConvolutionUtils.t23 * ConvolutionUtils.t18 * ConvolutionUtils.t20 / 8;
        return ConvolutionUtils.t26;
    }

    public static void printLaplaceOfGaussianKernel(
            int M, int N,
            double sigma) {
        float k[][] = getLaplaceOfGaussianKernel(M, N, sigma);
        //Mat.printKernel(k,"LaplaceOfGaussian"+k.length);
    }

    public static float[][] getLaplaceOfGaussianKernel(
            int M, int N,
            double sigma) {
        float k[][] = new float[M][N];
        int xc = M / 2;
        int yc = N / 2;
        for (int x = 0; x < k.length; x++)
            for (int y = 0; y < k[0].length; y++)
                k[x][y] = (float) laplaceOfGaussian(x, y, xc, yc, sigma);
        Mat2.normalize(k);
        return k;
    }

    public static float[][] getLaplacian9() {
        float k[][] = {
            {-1, -1, -1, -1, -1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1, -1, -1, -1, -1},
            {-1, -1, -1, 8, 8, 8, -1, -1, -1},
            {-1, -1, -1, 8, 8, 8, -1, -1, -1},
            {-1, -1, -1, 8, 8, 8, -1, -1, -1},
            {-1, -1, -1, -1, -1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1, -1, -1, -1, -1},
            {-1, -1, -1, -1, -1, -1, -1, -1, -1}};
        return k;
    }

    public static float[][] getHat13() {
        float k [][] = {
            {0, 0, 0, 0, 0, -1, -1, -1, 0, 0, 0, 0, 0},
            {0, 0, 0, -1, -1, -2, -2, -2, -1, -1, 0, 0, 0},
            {0, 0, -2, -2, -3, -3, -4, -3, -3, -2, -2, 0, 0},
            {0, -1, -2, -3, -3, -3, -2, -3, -3, -3, -2, -1, 0},
            {0, -1, -3, -3, -1, 4, 6, 4, -1, -3, -3, -1, 0},
            {-1, -2, -3, -3, 4, 14, 19, 14, 4, -3, -3, -2, -1},
            {-1, -2, -4, -2, 6, 19, 24, 19, 6, -2, -4, -2, -1},
            {-1, -2, -3, -3, 4, 14, 19, 14, 4, -3, -3, -2, -1},
            {0, -1, -3, -3, -1, 4, 6, 4, -1, -3, -3, -1, 0},
            {0, -1, -2, -3, -3, -3, -2, -3, -3, -3, -2, -1, 0},
            {0, 0, -2, -2, -3, -3, -4, -3, -3, -2, -2, 0, 0},
            {0, 0, 0, -1, -1, -2, -2, -2, -1, -1, 0, 0, 0},
            {0, 0, 0, 0, 0, -1, -1, -1, 0, 0, 0, 0, 0}};
        return k;
    }

    public static float[][] getLp1() {
        float k[][] = {
            {1, 1, 1},
            {1, 2, 1},
            {1, 1, 1}
        };
        Mat2.scale(k, 1 / 10.0);
        return k;
    }

    public static float[][] getLp2() {
        float k[][] = {
            {1, 1, 1},
            {1, 4, 1},
            {1, 1, 1}
        };
        Mat2.scale(k, 1 / 12.0);
        return k;
    }

    public static float[][] getLp3() {
        float k[][] = {
            {1, 1, 1},
            {1, 12, 1},
            {1, 1, 1}
        };
        Mat2.scale(k, 1 / 20.0);
        return k;
    }

    public static float[][] getGabor7() {
        float k[][] = {

            {137, 126, 116, 123, 127, 128, 0},
            {148, 168, 133, 104, 117, 127, 0},
            {124, 173, 223, 158, 99, 113, 0},
            {117, 111, 181, 255, 181, 111, 0},
            {126, 113, 99, 158, 223, 173, 0},
            {128, 127, 117, 104, 133, 168, 0},
            {0, 0, 0, 0, 0, 0, 0}
        };
        Mat2.normalize(k);
        return k;
    }

    public static float[][] getMean9() {
        float s = (float) 81.0;
        float k[][] = {
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s},
            {1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s, 1 / s}};
        return k;
    }

    public static float[][] getMean3() {
        float k[][] = {
            {0.11111111f, 0.11111111f, 0.11111111f},
            {0.11111111f, 0.11111111f, 0.11111111f},
            {0.11111111f, 0.11111111f, 0.11111111f}};
        return k;
    }

    public static short[][] getMedianOctagon5x5() {
        short k[][] = {
            {0, 1, 1, 1, 0},
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1},
            {0, 1, 1, 1, 0}};
        return k;
    }

    public static short[][] getMedianDiamond() {
        short k[][] = {
            {0, 0, 0, 1, 0, 0, 0},
            {0, 0, 1, 1, 1, 0, 0},
            {0, 1, 1, 1, 1, 1, 0},
            {1, 1, 1, 1, 1, 1, 1},
            {0, 1, 1, 1, 1, 1, 0},
            {0, 0, 1, 1, 1, 0, 0},
            {0, 0, 0, 1, 0, 0, 0}};
        return k;
    }

    public static short[][] getMedianCross3x3() {
        short k[][] = {
            {1, 1, 1},
            {1, 1, 1},
            {1, 1, 1}
        };
        return k;
    }

    public static short[][] getMedianCross5x5() {
        short k[][] = {
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1}};
        return k;
    }

    public static short[][] getMedianCross7x7() {
        short k[][] = {
            {0, 0, 0, 1, 0, 0, 0},
            {0, 0, 0, 1, 0, 0, 0},
            {0, 0, 0, 1, 0, 0, 0},
            {1, 1, 1, 1, 1, 1, 1},
            {0, 0, 0, 1, 0, 0, 0},
            {0, 0, 0, 1, 0, 0, 0},
            {0, 0, 0, 1, 0, 0, 0}};
        return k;
    }

    public static short[][] getMedianSquare7x7() {
        short k[][] = {
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1},
            {1, 1, 1, 1, 1, 1, 1}};
        return k;
    }

    public static float[][] getAverage3x3() {
        float k[][] = {
            {1, 1, 1},
            {1, 1, 1},
            {1, 1, 1}
        };
        Mat2.scale(k, 1 / 9.0);
        return k;
    }

    public static float[][] getHp1() {
        float k[][] = {
            {0, -1, 0},
            {-1, 10, -1},
            {0, -1, 0}
        };
        Mat2.normalize(k);
        return k;
    }

    public static float[][] getHp2() {
        float k[][] = {
            {0, -1, 0},
            {-1, 8, -1},
            {0, -1, 0}
        };
        Mat2.normalize(k);
        return k;
    }

    public static float[][] getHp3() {
        float k[][] = {
            {0, -1, 0},
            {-1, 5, -1},
            {0, -1, 0}
        };
        Mat2.normalize(k);
        return k;
    }

    public static float[][] getHp4() {
        float k[][] = {
            {1, -2, 1},
            {-2, 5, -2},
            {1, -2, 1}
        };
        return k;
    }

    public static float[][] getHp5() {
        float k[][] = {
            {-1, -1, -1},
            {-1, 9, -1},
            {-1, -1, -1}
        };
        return k;
    }

    public static void main(String args[]) {
        printMehrotraAndZhang();
    }

    public static void printMehrotraAndZhang() {
        Mat2.printKernel(getMehrotraAndZhangKernel(9, 9, 1), "Mehrotra and Zhang");
    }

    public static float[][] getMehrotraAndZhangKernel(
            int M, int N,
            double h) {
        float k[][] = new float[M][N];
        double deltaX = 1 / (M * 1.0);
        double deltaY = 1 / (N * 1.0);
        double startX = -1;
        double startY = -1;

        for (int x = 0; x < k.length; x++)
            for (int y = 0; y < k[0].length; y++) {
                k[x][y] = (float) mehrotraAndZhang(
                        startX + deltaX * x, startY + deltaY * y, h);
            }
        Mat2.normalize(k);
        return k;
    }

    /*
    -100/9/h**2*(1-(x**2+y**2)**(3/2)/h**3+7.5*(x**2+y**2)**(3/2)/h**3*ln((x**2+
y**2)**(1/2)/h))
can be input into maple, optimized and the following will result.
This is the core optimal detector  described by Mehrotra and Zhan on pp6
of Graphical Models and Image Processing, v58, No. 1, Jan. 1996.
    */
    public static double mehrotraAndZhang(double x, double y, double h) {
        ConvolutionUtils.t1 = h * h;
        ConvolutionUtils.t3 = x * x;
        ConvolutionUtils.t4 = y * y;
        ConvolutionUtils.t5 = ConvolutionUtils.t3 + ConvolutionUtils.t4;
        ConvolutionUtils.t6 = Math.sqrt(ConvolutionUtils.t5);
        ConvolutionUtils.t7 = ConvolutionUtils.t6 * ConvolutionUtils.t5;
        ConvolutionUtils.t9 = 1 / ConvolutionUtils.t1 / h;
        return -100.0 / 9.0 / ConvolutionUtils.t1 *
                (1.0 - ConvolutionUtils.t7 * ConvolutionUtils.t9 + 0.75E1 * ConvolutionUtils.t7 * ConvolutionUtils.t9 * Math.log(ConvolutionUtils.t6 / h));
    }
}